<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\HeadGroup;
use App\Models\Menu;
use App\Models\Organisation;
use Exception;
use App\Services\PermissionService;
use App\Models\AccountingYear;
use App\Models\Head;
use App\Traits\CommonTrait;
use Illuminate\Support\Facades\Validator;


class HeadController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    protected $currentOrgId;
    use CommonTrait;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->menuId = Menu::where('route', 'head.index')->value('id');
        $this->currentYear = AccountingYear::current();
        $this->currentOrgId =  auth()->user()->organisation_id;
    }

    public function index()
    {
        return view('head.index');
    }

    public function list(Request $request)
    {
        try {
            $search = $request->input('search');
            $page   = max(1, (int) $request->input('page', 1));
            $size   = max(1, (int) $request->input('size', 10));

            $query = Head::query()
                ->leftJoin('head_groups', 'heads.head_group_id', '=', 'head_groups.id')
                ->select('heads.*', 'head_groups.name as groupname')
                ->when($search, function ($q) use ($search) {
                    $q->where('heads.name', 'like', "%{$search}%");
                });

            $query->orderBy('sort_order', 'asc')
                ->orderBy('heads.id', 'desc');

            $result = $query->paginate($size, ['*'], 'page', $page);

            return response()->json([
                'data'         => $result->items(),
                'current_page' => $result->currentPage(),
                'last_page'    => $result->lastPage(),
                'per_page'     => $result->perPage(),
                'total'        => $result->total(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'line'  => $e->getLine(),
                'file'  => $e->getFile()
            ], 500);
        }
    }
    public function delete(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:heads,id'
        ]);

        try {
            $head = Head::find($request->id);


            if (!empty($head) && $head->delete()) {
                return response()->json([
                    'status' => true,
                    'message' => 'Deleted successfully',
                    'data' => $head,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Failed to delete. Please try again.',
                ], 500);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }
    public function changeStatus(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:heads,id',
            'new_status' => 'required|in:active,inactive'
        ]);

        try {

            $head = Head::findOrFail($request->id);


            $head->status = $request->new_status;
            $head->save();

            return response()->json([
                'status' => true,
                'message' => 'Status changed successfully',
                'data' => $head,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }


    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $organisations = Organisation::where('status', 'active')->get();
        $getAccountTye = $this->getAccountType('general');
        $headGroup = $this->getGroupHead();
        return view('head.create', ['organisations' => $organisations, 'divitions' => $getAccountTye, 'headGroup' => $headGroup, 'orgId' => $this->currentOrgId]);
    }

    public function store(Request $request)
    {
        try {

            $validator = Validator::make($request->all(), [
                'name'                    => 'required|string|max:255',
                'organisation_id'         => 'required|integer',
                'head_group_id'           => 'required|integer',
                'account_type_id'         => 'required|integer',
                'type'                    => 'nullable|string',
                'asset_or_liability_type' => 'nullable|string',
                'depriciation_type'       => 'nullable|in:flat,percentage',
                'depriciation_rate'       => 'nullable|string',
                'depriciation_tenure'     => 'nullable|string',
            ], [], [
                'name'                    => 'Head Name',
                'organisation_id'         => 'Organisation',
                'head_group_id'           => 'Head Group',
                'account_type_id'         => 'Division',
                'type'                    => 'Type',
                'asset_or_liability_type' => 'Asset / Liability Type',
                'depriciation_type'       => 'Depreciation Type',
                'depriciation_rate'       => 'Depreciation Rate',
                'depriciation_tenure'     => 'Depreciation Tenure',
            ]);

            if ($validator->fails()) {
                return redirect()
                    ->back()
                    ->withErrors($validator)
                    ->withInput();
            }
            $type = $request->type;
            $depriciationType = null;
            $assetOrLiabilityType = null;

            if ($request->has('rte_extention') && $request->rte_extention) {
                $depriciationType = $request->rte_extention === '%' ? 'percentage' : 'flat';
            }

            if ($request->has('asset_type')) {
                $assetOrLiabilityType = $request->asset_type;
            }
            if ($request->has('liability_type')) {
                $assetOrLiabilityType = $request->liability_type;
            }
            if (($request->has('liability_type') && !empty($request->liability_type)) || ($request->has('asset_type') && !empty($request->asset_type))) {
                $type = null;
            }

            $maxSortOrder = Head::where('organisation_id', $request->organisation_id)
                ->max('sort_order');
            $head = Head::create([
                'name'                      => trim($request->name),
                'organisation_id'           => $request->organisation_id,
                'head_group_id'             => $request->head_group_id,
                'account_type_id'           => $request->account_type_id,
                'type'                      => $type ?? null,
                'asset_or_liability_type'   => $assetOrLiabilityType,
                'depriciation_type'         => $depriciationType,
                'depriciation_rate'         => $request->depriciation_rate ?? null,
                'depriciation_tenure'       => $request->depriciation_tenure ?? null,
                'sort_order'                => ($maxSortOrder ?? 0) + 1,
                'status'                    => 'active',
            ]);
            return redirect()
                ->route('heads.index')
                ->with('success', 'Head "' . $head->name . '" created successfully.');
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Something went wrong while creating the Head. Please try again.');
        }
    }



    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        try {
            $decodedId = base64_decode($id);
            if (!is_numeric((int)$decodedId)) {
                abort(404, 'Invalid ID');
            }
            $head = Head::findOrFail($decodedId);
            $organisations = Organisation::where('status', 'active')->get();
            $getAccountTye = $this->getAccountType('general');
            $headGroup = $this->getGroupHead();
            return view('head.edit', [
                'head' => $head,
                'organisations' => $organisations,
                'divitions' => $getAccountTye,
                'headGroup' => $headGroup,
                'orgId' => $this->currentOrgId
            ]);
        } catch (\Exception $e) {
            abort(404, 'Invalid or corrupted ID');
        }
    }

    public function update(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'id'                        => 'required|exists:heads,id',
                'name'                      => 'required|string|max:255',
                'organisation_id'           => 'required|integer',
                'head_group_id'             => 'required|integer',
                'account_type_id'           => 'required|integer',
                'type'                      => 'nullable|string',
                'asset_or_liability_type'   => 'nullable|string',
                'depriciation_type'         => 'nullable|in:flat,percentage',
                'depriciation_rate'         => 'nullable|string',
                'depriciation_tenure'       => 'nullable|string',
            ], [], [
                'name'                    => 'Head Name',
                'organisation_id'         => 'Organisation',
                'head_group_id'           => 'Head Group',
                'account_type_id'         => 'Division',
                'type'                    => 'Type',
                'asset_or_liability_type' => 'Asset / Liability Type',
                'depriciation_type'       => 'Depreciation Type',
                'depriciation_rate'       => 'Depreciation Rate',
                'depriciation_tenure'     => 'Depreciation Tenure',
            ]);
            if ($validator->fails()) {
                return redirect()
                    ->back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $head = Head::findOrFail($request->id);

            $type = $request->type;
            $depriciationType = null;
            $assetOrLiabilityType = null;

            if ($request->has('rte_extention') && $request->rte_extention) {
                $depriciationType = $request->rte_extention === '%' ? 'percentage' : 'flat';
            }

            if ($request->has('asset_type')) {
                $assetOrLiabilityType = $request->asset_type;
            }
            if ($request->has('liability_type')) {
                $assetOrLiabilityType = $request->liability_type;
            }
            if (($request->has('liability_type') && !empty($request->liability_type)) || ($request->has('asset_type') && !empty($request->asset_type))) {
                $type = null;
            }

            $head->update([
                'name'                      => trim($request->name),
                'organisation_id'           => $request->organisation_id,
                'head_group_id'             => $request->head_group_id,
                'account_type_id'           => $request->account_type_id,
                'type'                      => $type ?? null,
                'asset_or_liability_type'   => $assetOrLiabilityType,
                'depriciation_type'         => $depriciationType,
                'depriciation_rate'         => $request->depriciation_rate ?? null,
                'depriciation_tenure'       => $request->depriciation_tenure ?? null,
            ]);

            return redirect()
                ->route('heads.index')
                ->with('success', 'Head "' . $head->name . '" Updated successfully.');
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Something went wrong while updating the Head. Please try again.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    // app/Http/Controllers/HeadController.php


    public function updateOrder(Request $request)
    {
        try {
            $items = $request->input('items'); // array of ID in new order

            foreach ($items as $index => $id) {
                Head::where('id', $id)->update([
                    'sort_order' => $index + 1
                ]);
            }

            return response()->json(['status' => 'success']);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'line'  => $e->getLine(),
                'file'  => $e->getFile()
            ], 500);
        }
    }
}
