<?php

namespace App\Http\Controllers;

use App\Models\AccountingYear;
use App\Models\AccountType;
use App\Models\ExpenseHeadGroup;
use App\Models\Menu;
use App\Models\Organisation;
use App\Services\PermissionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Traits\CommonTrait;
use Illuminate\Validation\Rule;

class ExpenseHeadController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    use CommonTrait;


    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->menuId = Menu::whereRaw('TRIM(route) = ?', ['expensehead.index'])->value('id');
        $this->currentYear = AccountingYear::current();
    }

    public function index()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Expense Head Records.');
        }
        $organisations = Organisation::where('status', 'active')->get();

        return view('auth.expenseHead.index', compact('organisations'));
    }

    public function getExpenseHeadDetails(Request $request)
    {
        try {
            if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
                abort(403, 'You do not have read access to Expense Heads.');
            }

            $search = $request->input('search');
            $organisation_id = $request->input('organisation_id');
            $page   = max(1, (int) $request->input('page', 1));
            $size   = max(1, (int) $request->input('size', 10));

            $sortField = $request->input('sorters.0.field', 'expense_head_id');
            $sortOrder = $request->input('sorters.0.dir', 'desc');
            $status = 'active';


            $query = \App\Models\ExpenseHead::query()
                    ->leftJoin('account_types', 'account_types.account_type_id', '=', 'expense_heads.account_type_id')
                    ->leftJoin('expense_head_groups', 'expense_head_groups.expense_head_group_id', '=', 'expense_heads.expense_head_group_id')
                    ->select(
                        'expense_heads.*',
                        'account_types.account_type_Name as account_type_name',
                        'expense_head_groups.expense_head_group_Name as group_name'
                    )
                ->when($organisation_id, fn($q) => $q->where('expense_heads.organisation_id', $organisation_id))
                ->when($search, function ($q) use ($search) {
                    $q->where(function ($query) use ($search) {
                        $query->where('expense_heads.expense_head_name', 'like', "%$search%")
                            ->orWhere('expense_head_groups.expense_head_group_Name', 'like', "%$search%");
                    });
                })
                ->when(isset($status), fn($q) => $q->where('expense_heads.status', $status)) // 👈 added status condition
                ->orderBy($sortField, $sortOrder);


            $result = $query->paginate($size, ['*'], '', $page);

            return response()->json([
                'data'         => $result->items(),
                'current_page' => $result->currentPage(),
                'last_page'    => $result->lastPage(),
                'per_page'     => $result->perPage(),
                'total'        => $result->total(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'line'  => $e->getLine(),
                'file'  => $e->getFile()
            ], 500);
        }
    }

    public function create()
    {
        $accountTypes = $this->getAccountType('general');
        $expenseHeadGroups = ExpenseHeadGroup::where('status', 'active')->get();
        $organisations = Organisation::where('status', 'active')->get();
        return view('auth.expenseHead.create', compact('accountTypes', 'expenseHeadGroups', 'organisations'));
    }

    public function store(Request $request)
    {
        try {

            // Conditional validation rules
            $rules = [
                'expense_head_name' => [
                    'required',
                    'string',
                    'max:255',
                    Rule::unique('expense_heads')->where(function ($query) use ($request) {
                        return $query->where('organisation_id', $request->organisation_id)
                                    ->where('expense_head_group_id', $request->expense_head_group_id)
                                    ->where('status', 'active');
                    }),
                ],
                'account_type_id' => 'nullable|integer',
                'organisation_id' => 'required|integer',
                'expense_head_group_id' => 'nullable|integer',
                'expense_head_group_opening_balance' => 'nullable|numeric',
            ];

            // Add asset fields if group_id == 3
            if ($request->expense_head_group_id == 3) {
                $rules['head_type'] = 'nullable';
                $rules['depriciation_rate'] = 'nullable';
                $rules['depriciation_tenure'] = 'nullable';
            }

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            // Insert record
        $depriciation_tenure = !empty($request->depriciation_rate)
            ? ($request->depriciation_tenure ?? 12)
            : ($request->depriciation_tenure ?? null);

        $id = DB::table('expense_heads')->insertGetId([
            'expense_head_name' => $request->expense_head_name,
            'organisation_id' => $request->organisation_id,
            'account_type_id' => $request->account_type_id ?? null,
            'expense_head_group_id' => $request->expense_head_group_id ?? null,
            'expense_head_group_opening_balance' => $request->expense_head_group_opening_balance ?? 0,
            'head_type' => $request->head_type ?? null,
            'depriciation_rate' => $request->depriciation_rate ?? null,
            'depriciation_tenure' => $depriciation_tenure,
            'asset_opening_balance'=> $request->asset_opening_balance ?? null,
            'created_at' => now(),
            'updated_at' => now(),
        ]);


            return response()->json([
                'success' => true,
                'message' => 'Expense head added successfully.',
                'new_id' => $id,
                'name' => $request->expense_head_name
            ], 201);

        } catch (\Exception $e) {
            dd($e);
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong!',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show(string $id)
    {
        //
    }

    public function edit(string $id)
    {
        //
    }

    public function update(Request $request, string $id)
    {
        //
    }

    public function destroy(string $id)
    {
        //
    }
}
