<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use App\Models\Donor;
use App\Models\Head;
use App\Models\AccountType;
use App\Models\PaymentType;
use App\Traits\CommonTrait;
use App\Models\Organisation;
use App\Models\AccountingYear;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\PermissionService;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class DonorController extends Controller
{
    use CommonTrait;
    protected $permissionService, $menuId, $currentYear, $currentOrgId, $currentOrgName, $currentMappingTable, $currentJournalTransactionsTable;
    public $filterableName = '';

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;

        $this->menuId = Menu::where('route', 'donor.index')->value('id');
        $this->currentYear = AccountingYear::current();
        $this->currentOrgId =  Auth::user()->organisation_id;
        $this->currentOrgName = $this->getOrganisationName($this->currentOrgId);
        $this->currentJournalTransactionsTable = $this->getYearPrefixedTables()['journal_transactions'];
        $this->currentMappingTable = $this->getYearPrefixedTables()['donor_transaction_record'];

        if (!$this->currentOrgId) {
            return redirect()->back()->with('error', 'Please make an organisation as current before proceeding.');
        }
    }

    public function index()
    {
        $organisations = Organisation::where('status', 'active')->get();
        return view('donor.index', ['organisations' => $organisations]);
    }

    public function create()
    {
        return view('donor.create', [
            'organisationsName' => $this->currentOrgName,
            'accountTypes' => $this->getAccountTypesByOrg($this->currentOrgId),
            'orgId' => $this->currentOrgId
        ]);
    }
    public function editDonor($id)
    {
        $donorId = base64_decode($id);
        if (!ctype_digit((string) $donorId)) {
            abort(404);
        }
        $donor = Donor::findOrFail($donorId)->toArray();
        return view('donor.edit', [
            'donor' => $donor,
            'organisationsName' => $this->currentOrgName,
            'accountTypes' => $this->getAccountTypesByOrg($this->currentOrgId),
            'orgId' => $this->currentOrgId
        ]);
    }
    public function donationStore(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'members'                => 'required|array|min:1',
            'members.*'              => 'required|exists:donors,donor_id',

            'income_heads'           => 'required|array|min:1',
            'income_heads.*'         => 'required|exists:heads,id',

            'amounts'                => 'required|array|min:1',
            'amounts.*'              => 'required|numeric|min:1',

            'payment_type_id'        => 'required|array|min:1',
            'payment_type_id.*'      => 'required|exists:payment_types,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                "status"    => false,
                "message"   => $validator->errors(),
            ], 422);
        }

        $members        = $request->members;
        $incomeHeads    = $request->income_heads;
        $amounts        = $request->amounts;
        $paymentTypes   = $request->payment_type_id;

        $donations = [];

        foreach ($members as $index => $memberId) {
            $donations[] = [
                'donorId'       => $memberId,
                'headId'         => $incomeHeads[$index],
                'amount'          => $amounts[$index],
                'paymentType'   => $paymentTypes[$index],
                'date'          => date("Y-m-d")
            ];
        }
        $errorMessage = 'Failed to add donation where ';
        $allInserted = true;
        foreach ($donations as $d) {
            $inserted = $this->transactionStoreHelper(false,  $d, null, false);
            $inserted = $inserted->getData(true);
            if (!$inserted['status']) {
                $allInserted = false;
                $errorMessage .= '[ Amount : ' . $inserted['error'] . '] ';
            }
        }
        return response()->json([
            "status"    => $allInserted,
            "message"   => $allInserted ? "Successdully Added." : $errorMessage,
        ], 200);
    }
    public function memberDonationCreate($id)
    {
        $donorId = base64_decode($id);
        $heads = $this->getHeadsByGroup('General Income', false);
        $paymentTypes = $this->getPaymentTypes();
        $donor = Donor::where('donor_id', $donorId)->first();
        if (!$donor) {
            return redirect()->back()->with('error', 'Donor not found');
        }
        return view(
            'donor.memberDonation',
            [
                'donor' => $donor,
                'heads' => $heads,
                'paymentTypes' => $paymentTypes
            ]
        );
    }
    public function importForm()
    {
        $accountTypes = AccountType::where('status', 'active')->get()->toArray();
        return view('donor.bulkUploadForm', ['orgId' => $this->currentOrgId, 'organisationsName' => $this->currentOrgName, 'accountTypes' => $accountTypes]);
    }
    public function memberDonationList()
    {
        $heads  = $this->getHeadsByGroup('General Income');
        $paymentTypes  = $this->getPaymentTypes();

        $donors = Donor::where('donor_status', 'active')->where('donor_type', 'members')->where('organisation_id', $this->currentOrgId)->get();
        $rows = DB::table('donors as d')
            ->leftJoin($this->currentMappingTable . ' as t', 't.donor_id', '=', 'd.donor_id')
            ->leftJoin($this->currentJournalTransactionsTable . ' as j', 'j.id', '=', 't.journal_id')
            ->leftJoin('heads as h', function ($join) {
                $join->on('h.id', '=', 'j.head_id')
                    ->where('h.head_group_id', 7)
                    ->where('h.status', 'active');
            })
            ->where('d.organisation_id', $this->currentOrgId)
            ->select(
                'd.donor_id',
                'd.donor_name',
                'd.account_type_id',
                'h.id as head_id',
                'h.head_group_id as head_group_id',
                'h.name as head_name',
                't.paid_amount',
                't.flag'
            )
            ->orderBy('d.donor_id')
            ->orderBy('h.sort_order')
            ->get();

        $transactionsData = $this->formatDonorTransactions($rows);
        $totalAmount = array_sum(array_column($transactionsData, 'totalPurchase'));

        return  view('donor.subscription.list', [
            'transactionsData' => $transactionsData,
            'heads' => $heads,
            'paymentTypes' => $paymentTypes,
            'donors' => $donors,
            'totalAmount'  => $totalAmount
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'donor_type' => 'required|in:members,foreign_members,non_members',
            'donor_name' => 'required|string|max:255',
            'donor_folio_no' => 'nullable',
            'account_type_id' => 'nullable',
            'donor_address' => 'nullable',
            'donor_phone_number' => 'nullable|string|max:20',
        ]);

        $donor = Donor::create([
            'organisation_id' => $this->currentOrgId,
            'donor_name' => $validated['donor_name'],
            'donor_type' => $validated['donor_type'],
            'account_type_id' => $validated['account_type_id'] ?? null,
            'donor_phone_number' => $validated['donor_phone_number'] ?? null,
            'donor_folio_no' =>  $validated['donor_folio_no'] ?? null,
            'donor_address' =>  $validated['donor_address'] ?? null
        ]);
        if ($donor) {
            return redirect()->route('donor.index')->with('success', 'Donor added successfully!');
        }
        return redirect()->route('donor.index')->with('error', 'Something went wrong. Please try again later.');
    }
    public function updateDonor(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer|min:1|exists:donors,donor_id',
            'donor_type' => 'required|in:members,foreign_members,non_members',
            'donor_name' => 'required|string|max:255',
            'donor_folio_no' => 'nullable',
            'account_type_id' => 'nullable',
            'donor_address' => 'nullable',
            'donor_phone_number' => 'nullable|string|max:20',
        ]);

        $donor = Donor::findOrFail($request->id);
        $donor = $donor->update([
            'donor_name'             => $validated['donor_name'],
            'donor_type'             => $validated['donor_type'],
            'account_type_id'        => $validated['account_type_id'] ?? null,
            'donor_folio_no'         => $validated['donor_folio_no'] ?? null,
            'donor_address'          => $validated['donor_address'] ?? null,
        ]);
        if ($donor) {
            return redirect()->route('donor.index')->with('success', 'Donor updated successfully!');
        }
        return redirect()->route('donor.index')->with('error', 'Something went wrong. Please try again later.');
    }

    public function getDonorDetails(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Donor Section.');
        }
        $organisation_id = !empty($request->input('organisation_id')) ? $request->input('organisation_id') : null;
        $search = trim($request->input('search'));
        $amount_flag = $request->input('amount_flag');

        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'donor_id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = Donor::query()
            ->when($amount_flag, function ($q) use ($amount_flag) {
                $q->where('amount_flag', $amount_flag);
            })
            ->when($search, function ($q) use ($search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('donor_name', 'like', "%{$search}%")
                        ->orWhere('amount_flag', 'like', "%{$search}%");
                });
            })
            ->when($organisation_id, function ($q) use ($organisation_id) {
                $q->where('organisation_id', $organisation_id);
            })
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);

        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function checkDonorIncome($donorId)
    {
        $income = DB::table($this->currentJournalTransactionsTable)
            ->where('donor_id', $donorId)
            ->where(['transaction_type' => 'income', 'headGroupId' => '7'])
            ->where('financial_year_id', $this->currentYear->accounting_year_id)
            ->orderByDesc('created_at')
            ->first();

        if ($income) {
            return response()->json([
                'exists' => true,
                'data'   => $income
            ]);
        }

        return response()->json(['exists' => false]);
    }

    public function importCSV(Request $request)
    {
        try {
            $request->validate([
                'csv_file'        => 'required|mimes:csv,txt',
                'account_type_id' => 'nullable|integer',
            ]);

            $donorType = 'members';
            $organisation_id = $this->currentOrgId;
            $account_type_id = $request->account_type_id;

            $file = $request->file('csv_file');
            $insertData = [];
            $rowCount = 0;
            $duplicateFolioNos = [];

            if (($handle = fopen($file->getRealPath(), 'r')) !== false) {
                $header = fgetcsv($handle, 1000, ',');
                $header = array_map('trim', $header);
                $header = array_filter($header, fn($col) => strtolower($col) !== 'donor_type');

                while (($row = fgetcsv($handle, 1000, ',')) !== false) {
                    $row = array_map('trim', $row);

                    if (count($row) == count($header)) {
                        $rowData = array_combine($header, $row);
                        $rowData['donor_type'] = $donorType;
                        $rowData['organisation_id'] = $organisation_id;
                        $rowData['account_type_id'] = $account_type_id;

                        $existing = Donor::where('donor_folio_no', $rowData['donor_folio_no'])->first();
                        if ($existing) {
                            $duplicateFolioNos[] = $rowData['donor_folio_no'];
                            continue;
                        }
                        $insertData[] = $rowData;
                        $rowCount = $rowCount + 1;
                    } else {
                        throw new \Exception(
                            'CSV row column count mismatch at row ' . ($rowCount + 2)
                        );
                    }
                }
                fclose($handle);
            }

            if (count($insertData) > 0) {
                foreach (array_chunk($insertData, 500) as $chunk) {
                    Donor::insert($chunk);
                }
            }

            $message = "$rowCount rows imported successfully!";
            if (count($duplicateFolioNos) > 0) {
                $message .= " Skipped duplicates: " . implode(', ', $duplicateFolioNos);
            }

            return back()->with('success', $message);
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function checkDonationExists(Request $request)
    {
        $donorId = $request->input('donor_id');
        $headId = $request->input('income_head_id');

        $exists = DB::table('incomes')
            ->where('donor_id', $donorId)
            ->where('income_head', $headId)
            ->exists();

        return response()->json(['exists' => $exists]);
    }

    public function downloadSampleCSV()
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="sample_donors.csv"',
        ];

        $columns = ['donor_name', 'donor_folio_no', 'donor_phone_number', 'donor_pan_number', 'donor_address'];

        $callback = function () use ($columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            fputcsv($file, ['member 1', 'FOL123', '9876543210', 'ABCDE1234F', 'ABCDE1234F']);

            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }

    public function submitTransaction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'headId'        => 'required|exists:heads,id',
            'headName'      => 'required|exists:heads,name',
            'headGroupId'   => 'required|exists:head_groups,id',
            'donorId'       => 'required|exists:donors,donor_id',
            'accountTypeId' => 'nullable|exists:account_types,account_type_id',
            'date'          => 'required|date',
            'paymentType'   => 'required|exists:payment_types,id',
            'amount'        => 'required|numeric|min:1',
            'receiptNo'     => 'required|string|max:191',
            'narration'     => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                "status" => true,
                "msg" => "Validation Errors.",
                'errors'  => $validator->errors(),
            ], 422);
        }

        $data = $validator->validated();

        $flagData = [];
        $isSubscription = $request['isSubscription'];
        $isCompleted = false;
        $isReadyToStoreSubscription =  $isSubscription === "true" ? $this->handleSubscriptionData($data) : [];

        if (!empty($isReadyToStoreSubscription) && $isReadyToStoreSubscription['status'] == false) {
            return json_encode([
                'status' => false,
                'msg'   => $isReadyToStoreSubscription['msg']
            ]);
        }
        if (!empty($isReadyToStoreSubscription) && $isReadyToStoreSubscription['status'] != false) {
            $isCompleted = $isReadyToStoreSubscription['isCompleted'];
            $flagData['member_fee'] = $isReadyToStoreSubscription['member_fee'];
            $flagData['contingent_fee'] = $isReadyToStoreSubscription['contingent_fee'];
        }

        return $this->transactionStoreHelper($isSubscription,  $data, $flagData, $isCompleted);
    }
    private function transactionStoreHelper($isSubscription,  $data, $flagData, $isCompleted)
    {
        DB::beginTransaction();
        try {
            $journalId = DB::table($this->currentJournalTransactionsTable)->insertGetId([
                'organisation_id'       => $this->currentOrgId,
                'account_type_id'       => $data['accountTypeId'] ?? null,
                'head_id'               => $data['headId'],
                'head_group_id'         => $data['headGroupId'] ?? null,
                'user_id'               => auth()->id(),
                'transaction_amount'    => $data['amount'],
                'payment_type_id'       => $data['paymentType'],
                'transaction_date'      => $data['date'] ?? null,
                'transaction_reference' => $data['transaction_reference'] ?? null,
                'transaction_narration' => $data['narration'] ?? null,
                'money_receipt_no'      => $data['receiptNo'] ?? null,
                'file_path'             => '',
                'transcation_type'      => 'income',
                'paid_by'               => 'donor',
                'donor_id'              => $data['donorId'],
                'created_at'            => now(),
                'updated_at'            => now(),
            ]);
            $mappingTable = false;
            if ($journalId) {
                $mappingTable = DB::table($this->currentMappingTable)->insert([
                    'journal_id'     => $journalId,
                    'donor_id'       => $data['donorId'],
                    'paid_amount'    => $data['amount'],
                    'income_expense' => 'income',
                    'dr_cr'          => 'dr',
                    'flag'           => $isSubscription === "true" ? json_encode($flagData) : json_encode($data['headName'] ?? ''),
                    'created_at'     => now(),
                    'updated_at'     => now(),
                ]);
            }

            if ($isSubscription && $mappingTable) {
                Donor::where('donor_id', $data['donorId'])
                    ->update([
                        'amount_flag' => $isCompleted ? 'paid' : 'not_paid',
                        'updated_at'  => now(),
                    ]);
            }
            DB::commit();
            return response()->json([
                'status' => true,
                'msg' => 'Transaction recorded successfully'
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status' => false,
                'msg'     => 'Failed to record transaction.',
                'data'    => $data,
                'error'   => $e->getMessage(),
            ], 500);
        }
    }
    private function formatDonorTransactions($rows): array
    {
        $allHeads = Head::select('id', 'name', 'head_group_id')
            ->where(['head_group_id' => 7, 'status' => 'active'])
            ->orderBy('sort_order')
            ->get();
        $retData = $this->buildDonorWiseData($rows, $allHeads);
        return $retData;
    }
    private function handleSubscriptionData($data)
    {
        $paymentAmount = round((float) $data['amount'], 2);

        $contingentCap   = 100;
        $subscriptionCap = 400;

        $existingRecords = DB::table($this->currentMappingTable . ' as m')
            ->join($this->currentJournalTransactionsTable . ' as j', 'm.journal_id', '=', 'j.id')
            ->where('j.donor_id', $data['donorId'])
            ->where('j.head_id', $data['headId'])
            ->select('m.*')
            ->get();

        $contingentPaid = 0;
        $subscriptionPaid = 0;

        foreach ($existingRecords as $record) {
            $flag = json_decode($record->flag, true);
            if (is_array($flag)) {
                $contingentPaid   += (float) ($flag['contingent_fee'] ?? 0);
                $subscriptionPaid += (float) ($flag['member_fee'] ?? 0);
            }
        }

        $contingentPaid   = round($contingentPaid, 2);
        $subscriptionPaid = round($subscriptionPaid, 2);

        $remainingContingent   = max($contingentCap - $contingentPaid, 0);
        $remainingSubscription = max($subscriptionCap - $subscriptionPaid, 0);
        $totalRemaining        = $remainingContingent + $remainingSubscription;

        if ($totalRemaining <= 0 || $paymentAmount > $totalRemaining) {
            $falseArray = [
                'status' => false
            ];
            if ($totalRemaining <= 0) $falseArray['msg'] = 'Amount already fully paid.';
            if ($paymentAmount > $totalRemaining) $falseArray['msg'] = 'Total amount exceeds Rs. 500';
        }
        $contingentContribution   = min($paymentAmount, $remainingContingent);
        $subscriptionContribution = round($paymentAmount - $contingentContribution, 2);
        $contingentContribution   = round($contingentContribution, 2);

        $newContingentTotal   = round($contingentPaid + $contingentContribution, 2);
        $newSubscriptionTotal = round($subscriptionPaid + $subscriptionContribution, 2);

        $isCompleted = $newContingentTotal >= $contingentCap
            && $newSubscriptionTotal >= $subscriptionCap;

        return [
            'status'         => true,
            'msg'            => 'Ready to store',
            'isCompleted'    =>  $isCompleted,
            'member_fee'     => $subscriptionContribution,
            'contingent_fee' => $contingentContribution,
        ];
    }
    private function buildDonorWiseData($rows, $allHeads): array
    {
        $data = $rows->groupBy('donor_id')->map(function ($donorRows) use ($allHeads) {
            return $this->buildSingleDonorData($donorRows, $allHeads);
        })->values()->toArray();
        return $data;
    }
    private function buildSingleDonorData($donorRows, $allHeads): array
    {
        $headWiseData = $this->buildHeadWiseData($donorRows, $allHeads);

        return [
            'donor_id'         => $donorRows->first()->donor_id,
            'account_type_id'  => $donorRows->first()->account_type_id,
            'name'             => $donorRows->first()->donor_name,
            'totalPurchase'    => $donorRows->sum('paid_amount'),
            'transactions'     => $headWiseData,
        ];
    }
    private function buildHeadWiseData($donorRows, $allHeads): array
    {
        $headWiseData = [];
        foreach ($allHeads as $head) {
            $headWiseData[] = $this->buildSingleHeadData($donorRows, $head);
        }
        return $headWiseData;
    }
    private function buildSingleHeadData($donorRows, $head): array
    {
        $headRows = $donorRows->where('head_id', $head->id);
        $data = [];
        if ($this->isSpecialHead($head, $headRows)) {
            $data = $this->buildSpecialHeadData($head, $headRows);
        } else {
            $data = $this->buildNormalHeadData($head, $headRows);
        }
        return $data;
    }
    private function isSpecialHead($head, $headRows): bool
    {
        return $head->id == 13 && $headRows->isNotEmpty();
    }
    private function buildSpecialHeadData($head, $headRows): array
    {
        $data = json_decode($headRows->pluck('flag'), true);
        $transactions = array_reduce($data, function ($carry, $e) {
            $e = json_decode($e, true);
            $carry['member_fee'] += $e['member_fee'] ?? 0;
            $carry['contingent_fee'] += $e['contingent_fee'] ?? 0;
            return $carry;
        }, [
            'member_fee' => 0,
            'contingent_fee' => 0,
        ]);
        return [
            'headName'     => $head->name,
            'headId'       => $head->id,
            'headGroupId'  => $head->head_group_id,
            'total'        => $headRows->sum('paid_amount'),
            'transactions' => $transactions,
        ];
    }

    private function buildNormalHeadData($head, $headRows): array
    {
        $transactions = $this->buildTransactions($headRows);

        return [
            'headName'     => $head->name,
            'headId'       => $head->id,
            'headGroupId'  => $head->head_group_id,
            'total'        => array_sum(array_column($transactions, 'amount')),
            'transactions' => $transactions,
        ];
    }
    private function buildTransactions($headRows): array
    {
        $data = $headRows->map(function ($row) {
            return [
                'amount' => (float) $row->paid_amount,
                'type'   => is_string($row->flag) ? 'cash' : 'UPI',
                'date'   => now()->format('d-m-Y')
            ];
        })->toArray();
        return $data;
    }
}
