<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use App\Models\Asset;
use App\Models\IncomeHead;
use App\Models\AccountType;
use App\Traits\CommonTrait;
use App\Models\Organisation;
use Illuminate\Http\Request;
use App\Models\AccountingYear;
use Illuminate\Support\Facades\DB;
use App\Services\PermissionService;
use App\Models\Head;

class AssetController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    protected $tables;
    use  CommonTrait;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->menuId = Menu::where('route', 'asset.index')->value('id');
        $this->currentYear = AccountingYear::current();
        $this->tables = $this->getYearPrefixedTables();
        $this->assets_entry = $this->tables['assets_entry'];
        $this->asset_transaction_record = $this->tables['asset_transaction_record'];
    }

    public function index()
    {
        $organisations = Organisation::where('status', 'active')->get();
        return view('assets.index', compact('organisations'));
    }

    public function getAssetDetails(Request $request)
    {

        $search = $request->input('search');
        $organisation_id = $request->input('organisation_id');

        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'asset_id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = Asset::with('organisation', 'accountType')
            ->where('financial_year_id', $this->currentYear->accounting_year_id)
            ->when($organisation_id, fn($q) => $q->where('organisation_id', $organisation_id))
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('asset_name', 'like', "%{$search}%")
                        ->orWhere('asset_useful_life', 'like', "%{$search}%")
                        ->orWhereHas(
                            'organisation',
                            fn($q) =>
                            $q->where('organisation_name', 'like', "%{$search}%")
                        )
                        ->orWhereHas(
                            'accountType',
                            fn($q) =>
                            $q->where('account_type_name', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        $financialYears = AccountingYear::where('status', 'active')->get();
        $accountTypes = AccountType::where('status', 'active')->get();
        $organisations = Organisation::where('status', 'active')->get();
        // Get heads without division dependency (pass null to getHeads to get all heads)
        $heads = Head::where('status', 'active')->whereIn('asset_or_liability_type', [
        'financial_asset',
        'current_asset',
        'fixed_asset'
        ])->get();
        
        $banks = $this->getActiveBanks();
        return view('assets.create', compact('organisations', 'accountTypes', 'financialYears', 'heads', 'banks'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'organisation_id'       => 'required|integer',
            'account_type_id'       => 'nullable|integer',
            'head_id'               => 'nullable|integer',
            'bank_id'               => 'nullable|integer',
            'asset_name'            => 'required|string|max:255|unique:assets,asset_name',
            'asset_useful_life'     => 'nullable|integer',
            'purchase_date'         => 'required|date',
            'purchase_cost'         => 'required|numeric',
        ]);

        DB::beginTransaction();
        try {
            if ($this->assets_entry) {
                DB::table($this->assets_entry)->insert([
                    'asset_id'          => $id,
                    'organisation_id'   => $validated['organisation_id'],
                    'account_type_id'   => $validated['account_type_id'] ?? null,
                    'head_id'           => $validated['head_id'] ?? null,
                    'bank_id'           => $validated['bank_id'] ?? null,
                    'amount'            => $validated['purchase_cost'],
                    'purchase_date'     => $validated['purchase_date'],
                    'created_at'        => now(),
                    'updated_at'        => now(),
                ]);
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Asset added successfully.',
                'new_id'  => $id,
                'name'    => $validated['asset_name']
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

}
