<?php

namespace App\Http\Controllers;

use App\Models\AssetBroughtForward;
use App\Models\AccountingYear;
use App\Models\Organisation;
use Carbon\Carbon;
use Illuminate\Http\Request;


class AssetBroughtForwardController extends Controller
{
    protected $currentYear;
    protected $currentOrg;

    public function __construct()
    {
        $this->currentYear = AccountingYear::current();
        $this->currentOrg = Organisation::current();
    }

    public function index()
    {
        $today = Carbon::now();
        $year = $today->year;

        if ($today->month < 4) {
            $startYear = $year - 1;
            $endYear   = $year;
        } else {
            $startYear = $year;
            $endYear   = $year + 1;
        }

        $financialYearStart = Carbon::createFromDate($startYear, 4, 1)->toDateString();

        $recordExists = AssetBroughtForward::where('dated_as_on', $financialYearStart)->count();

        return view('assetBroughtForward.index', compact('recordExists'));
    }

    public function getAssetBroughtForwardDetails(Request $request)
    {
        $search = $request->input('search');
        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = AssetBroughtForward::with('financialYear')
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('total_asset_amount', 'like', "%{$search}%")
                        ->orWhereHas(
                            'financialYear',
                            fn($q) =>
                            $q->where('accounting_year_financial', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);

        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        $years = AccountingYear::where('status', 'active')->get();
        return view('assetBroughtForward.create', compact('years'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'accounting_year_id' => 'required|integer|exists:accounting_years,accounting_year_id',
            'total_asset_amount' => 'required|numeric|min:0',
            'transaction_date'   => 'required|date',
        ]);

        if (!auth()->check()) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'You must be logged in to perform this action.']);
        }

        if (!$this->currentOrg || !$this->currentOrg->organisation_id) {
            return redirect()->back()
                ->withInput()
                ->withErrors(
                    ['organisation_id' => 'Please select a current organisation before adding asset brought forward.'],
                    'orgCheck'
                );
        }

        try {
            $data = [
                'financial_year_id' => $validated['accounting_year_id'],
                'total_asset_amount' => $validated['total_asset_amount'],
                'user_id'           => auth()->user()->id,
                'dated_as_on'       => $validated['transaction_date'],
                'organisation_id'   => $this->currentOrg->organisation_id,
            ];

            $assetBroughtForward = AssetBroughtForward::create($data);

            return redirect()->route('asset.brought.forward.index')->with('success', 'Asset Brought Forward added successfully.');
        } catch (\Illuminate\Database\QueryException $e) {
            \Log::error('Asset Brought Forward Database Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request' => $request->all()
            ]);

            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'error' => 'Database error: ' . $e->getMessage()
                ]);
        } catch (\Exception $e) {
            \Log::error('Asset Brought Forward Store Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'request' => $request->all()
            ]);

            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'error' => 'Failed to save Asset Brought Forward. Please try again.'
                ]);
        }
    }

    public function edit(string $id)
    {
        $id = base64_decode($id);
        $assetBroughtForward = AssetBroughtForward::where('id', $id)->first();
        $years = AccountingYear::where('status', 'active')->get();

        return view('assetBroughtForward.edit', compact('assetBroughtForward', 'years'));
    }

    public function update(Request $request, string $id)
    {
        $validated = $request->validate([
            'accounting_year_id' => 'required|integer|exists:accounting_years,accounting_year_id',
            'total_asset_amount' => 'required|numeric|min:0',
            'transaction_date' => 'required|date',
        ]);

        $id = base64_decode($id);
        $assetBroughtForward = AssetBroughtForward::findOrFail($id);

        $assetBroughtForward->update([
            'financial_year_id' => $validated['accounting_year_id'] ?? $assetBroughtForward->financial_year_id,
            'total_asset_amount' => $validated['total_asset_amount'] ?? $assetBroughtForward->total_asset_amount,
            'user_id'   => auth()->user()->id ?? $assetBroughtForward->user_id,
            'dated_as_on' => $validated['transaction_date'] ?? $assetBroughtForward->dated_as_on
        ]);

        return redirect()->route('asset.brought.forward.index')->with('success', 'Asset Brought Forward updated successfully.');
    }

    public function destroy(string $id)
    {
        try {
            $id = base64_decode($id);
            $assetBroughtForward = AssetBroughtForward::findOrFail($id);
            $assetBroughtForward->delete();

            return redirect()->route('asset.brought.forward.index')->with('success', 'Asset Brought Forward deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to delete Asset Brought Forward.');
        }
    }
}
