<?php

namespace App\Jobs;

use App\Models\Student;
use App\Models\ParentModel;
use App\Models\StudentClassDetail;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use Throwable;
use Illuminate\Foundation\Bus\Dispatchable;
use PhpOffice\PhpSpreadsheet\Shared\Date as ExcelDate;

class StudentBulkUploadJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $timeout = 1200;
    public int $tries = 1;

    protected string $path;
    protected int $uploadId;
    protected array $headerMap;
    protected int $classId;
    protected int $sectionId;
    protected int $financialYearId;

    public function __construct(
        string $path,
        int $uploadId,
        array $headerMap,
        int $classId,
        int $sectionId,
        int $financialYearId
    ) {
        $this->path = $path;
        $this->uploadId = $uploadId;
        $this->headerMap = $headerMap;
        $this->classId = $classId;
        $this->sectionId = $sectionId;
        $this->financialYearId = $financialYearId;
    }

    public function handle(): void
    {
        $hasErrors = false;

        try {

            /* ================= LOAD EXCEL ================= */

            $fullPath = Storage::disk('local')->path($this->path);

            if (!file_exists($fullPath)) {
                throw new \Exception("File not found: {$fullPath}");
            }

            $rows = Excel::toArray([], $fullPath)[0];
            unset($rows[0]); // remove header row

            DB::table('student_bulk_uploads')
                ->where('id', $this->uploadId)
                ->update(['status' => 'running']);

            /* ================= PROCESS ROWS ================= */

            foreach ($rows as $index => $row) {

                $rowNumber = $index + 1;

                // ⛔ SKIP BLANK / JUNK EXCEL ROWS (VERY IMPORTANT)
                if ($this->isBlankRow($row)) {
                    continue; // DO NOT COUNT
                }

                $data = $this->mapRow($row);

                // Normalize phone
                // $data['father_phone'] = isset($data['father_phone'])
                //     ? (string) $data['father_phone']
                //     : null;
                
                $phone = explode('/',$data['father_phone']);

                $data['father_phone'] = $phone[0];
                if(count($data['father_phone'])>1){
                    $data['mother_phone'] = $phone[1];
                }
                // Convert dates
                $data['dob'] = $this->excelDateToDate($data['dob'] ?? null);
                $data['admission_date'] = $this->excelDateToDate($data['admission_date'] ?? null);

                $validator = $this->validator($data);

                if ($validator->fails()) {
                    $this->failRow($rowNumber, $data, $validator->errors()->first());
                    $hasErrors = true;
                    $this->increment();
                    continue;
                }

                // Duplicate student check
                $exists = Student::where('name', $data['name'])
                    ->whereHas('parent', fn ($q) =>
                        $q->where('father_phone', $data['father_phone'])
                    )->exists();

                if ($exists) {
                    $this->failRow($rowNumber, $data, 'Duplicate student');
                    $hasErrors = true;
                    $this->increment();
                    continue;
                }

                try {

                    DB::transaction(function () use ($data) {

                        $parent = ParentModel::create([
                            'father_name'   => $data['father_name'],
                            'mother_name'   => $data['mother_name'],
                            'father_phone' => $data['father_phone'],
                        ]);

                        // Admission No (lock safe)
                        $lastId = DB::table('students')->lockForUpdate()->max('id') ?? 0;
                        $admissionNo = 'ADM-' . date('Y') . '-' . str_pad($lastId + 1, 5, '0', STR_PAD_LEFT);

                        $student = Student::create([
                            'name'         => $data['name'],
                            'admission_no' => $admissionNo,
                            'dob'          => $data['dob'],
                            'gender'       => $data['gender'],
                            'category'     => $data['category'],
                            'address'      => $data['address'],
                            'parent_id'    => $parent->id,
                            'status'       => 1,
                        ]);

                        $maxRoll = StudentClassDetail::where([
                            'class_id' => $this->classId,
                            'section_id' => $this->sectionId,
                            'financial_year_id' => $this->financialYearId,
                            'status' => 1
                        ])->max('roll_no');

                        StudentClassDetail::create([
                            'student_id'        => $student->id,
                            'class_id'          => $this->classId,
                            'section_id'        => $this->sectionId,
                            'roll_no'           => $maxRoll ? $maxRoll + 1 : 1,
                            'financial_year_id' => $this->financialYearId,
                            'admission_date'    => $data['admission_date'],
                            'start_date'        => now(),
                            'status'            => 1,
                        ]);
                    });

                } catch (Throwable $e) {
                    $this->failRow($rowNumber, $data, $e->getMessage());
                    $hasErrors = true;
                }

                $this->increment();
            }

        } catch (Throwable $e) {

            $this->failRow(0, [], $e->getMessage());
            $hasErrors = true;

        } finally {

            DB::table('student_bulk_uploads')
                ->where('id', $this->uploadId)
                ->update([
                    'status' => $hasErrors
                        ? 'completed_with_errors'
                        : 'completed'
                ]);
        }
    }

    /* ================= JOB FAILED ================= */

    public function failed(Throwable $exception): void
    {
        DB::table('student_bulk_uploads')
            ->where('id', $this->uploadId)
            ->update([
                'status' => 'failed',
                'error_message' => substr($exception->getMessage(), 0, 1000),
            ]);
    }

    /* ================= HELPERS ================= */

    // 🔥 THIS FIXES YOUR BLANK ROW ISSUE
    private function isBlankRow(array $row): bool
    {
        foreach ($row as $value) {

            if (is_string($value)) {
                $value = trim(str_replace('`', '', $value));
            }

            if ($value !== null && $value !== '') {
                return false; // real data found
            }
        }

        return true;
    }

    private function mapRow(array $row): array
    {
        $mapped = [];

        foreach ($this->headerMap as $index => $field) {
            if ($field === 'sl_no') continue;

            $value = $row[$index] ?? null;

            if (is_string($value)) {
                $value = trim(str_replace('`', '', $value));
            }

            $mapped[$field] = $value === '' ? null : $value;
        }

        return $mapped;
    }

    private function validator(array $data)
    {
        return Validator::make($data, [
            'admission_date' => 'required|date',
            'name'           => 'required',
            'gender'         => 'required',
            'category'       => 'required',
            'address'        => 'required',
            'father_name'    => 'required',
            'mother_name'    => 'required',
            //'father_phone'   => 'required|regex:/^[0-9]{10}$/',
            'father_phone'   => 'required',
        ]);
    }

    private function increment(): void
    {
        DB::table('student_bulk_uploads')
            ->where('id', $this->uploadId)
            ->increment('processed_rows');
    }

    private function failRow(int $row, array $data, string $reason): void
    {
        DB::table('student_bulk_failed')->insert([
            'upload_id' => $this->uploadId,
            'row_number'=> $row,
            'data'      => json_encode($data),
            'reason'    => substr($reason, 0, 1000),
            'created_at'=> now(),
        ]);
    }

    private function excelDateToDate($value): ?string
    {
        if (empty($value)) return null;

        if (is_numeric($value)) {
            return ExcelDate::excelToDateTimeObject($value)->format('Y-m-d');
        }

        if (is_string($value) && strtotime($value)) {
            return date('Y-m-d', strtotime($value));
        }

        return null;
    }
}
