<?php
namespace App\Http\Controllers\Api;
use App\Http\Controllers\Controller;
use App\Http\Resources\HomepageResource;
use App\Http\Resources\PageResource;
use App\Http\Resources\ContinentResource;
use App\Http\Resources\LocationResource;
use App\Http\Resources\CategoryResource;
use App\Http\Resources\PackageResource;
use App\Http\Resources\CarCategoryResource;
use App\Http\Resources\CarResource;
use App\Http\Resources\CarRouteResource;

use Illuminate\Http\Request;

use App\Models\Banner;
use App\Models\SpecialPackage;
use App\Models\Package;
use App\Models\Location;
use App\Models\Page;
use App\Models\DealPackages;
use App\Models\TourService;
use App\Models\Car;
use App\Models\CarRoute;
use App\Models\PromotionalAd;

class SettingsController extends Controller {

    public function menu()
    {
        $packageSubQuery = \DB::table('packages')
            ->select('location_id as location_id')
            ->where('country_id', 1)
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->union(
                \DB::table('packages')
                    ->select('source_location_id as location_id')
                    ->where('country_id', 1)
                    ->where('is_active', 1)
                    ->where('is_deleted', '!=', 1)
            );

        $baseQuery = Location::whereIn('id', $packageSubQuery);

        $cities = (clone $baseQuery)
            ->distinct()
            ->get(['slug', 'name']);


        $menu = getRegionWiseCities($cities);

        $promotions = PromotionalAd::where('is_active', 1)
            ->whereIn('type', [1, 2])
            ->latest()
            ->get()
            ->makeHidden(['id', 'created_at', 'updated_at'])
            ->groupBy('type');

        // Now you can access:
        $national = $promotions->get(1)?->first();
        if($national){
            $national->banner_image = $national->banner_image ? storage_link($national->banner_image): null;
        }

        $international = $promotions->get(2)?->first();
        if($international){
            $international->banner_image = $international->banner_image ? storage_link($international->banner_image): null;
        }
            
        $limit     = 6; // default 10 
        $orderBy   = list_config()['order_by']; // default order by id
        $direction = list_config()['direction']; // default desc

        // Get all matching packages first
        $packages = Package::where('is_active', 1)
            ->where('type', 2) //  2 = international 
            ->where('is_top_trending', 1)
            ->where('is_deleted', '!=', 1)
            ->with(['location.details'])
            ->withAvg('reviews', 'rating')
            ->orderBy($orderBy, $direction)
            ->get();

        // Filter so that we keep only unique continent-country-location combos
        $unique = $packages->unique(function ($item) {
            return $item->continent_id . '-' . $item->country_id . '-' . $item->location_id;
        });

        // Take only required limit
        $result = $unique->pluck('location')->take($limit)->values();

        return response()->json([
            'india_mega_menu'         => $menu,
            'india_promotion'         => $national ? $national : null,
            'international_mega_menu' => LocationResource::collection($result),
            'international_promotion' => $international ? $international : null,

        ]);
    }


    public function home()
    {
        $page_details = Page::with('meta')
            ->where('id', 8)
            ->firstOrFail();

        $data = [
            'slider_banners'  => Banner::with('package')->where('is_active', 1)->where('is_static', 0)->where('is_deleted', 0)->get(),
            'single_banner'   => Banner::with('package')->where('is_active', 1)->where('is_static', 1)->where('is_deleted', 0)->first(),
            'special_package' => SpecialPackage::with('package.details')->where('is_active', 1)->where('is_deleted', 0)->first(),
            'tour_services'   => TourService::where('is_active', 1)->where('is_deleted', 0)->get(),
        ];

        return response()->json([
            'status'  => 'success',
            'message' => 'Details',
            'data'    => new HomepageResource([
                'extra' => $data,
                'page'  => $page_details
            ])
        ]);
    }


    public function international(Request $r)
    {
        $page_details = Page::with('faqs','meta')
            ->where('id', 1) // international page
            ->firstOrFail();

        $packages = Package::where('is_active', 1)
            ->where('type', 2) // International
            ->where('is_deleted', '!=', 1)
            ->with(['continent', 'category'])
            ->get();

        // 🔹 Extract unique continents from packages
        $continents = $packages
            ->pluck('continent')
            ->unique('id')
            ->values();

        // 🔹 Extract unique categories from packages
        $categories = $packages
            ->pluck('category')
            ->unique('id')
            ->values();

        $deal_packages = DealPackages::where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->with(['package.details','package.location.country.continent'])
            ->get()
            ->pluck('package')
            ->filter()
            ->values();

        return response()->json([
            'status'  => 'success',
            'message' => 'details',
            'data'    => [
                'details'    => new PageResource($page_details),
                'continents' => ContinentResource::collection($continents),
                'categories' => CategoryResource::collection($categories),
                'deal_packages' => PackageResource::collection($deal_packages)
            ],
        ], 200);
    }

    public function india(Request $r)
    {
        return $this->getSetting(1,3,$r);

    }

    public function srilanka(Request $r)
    {
        return $this->getSetting(10,4,$r);
        
    }

    private function getSetting($country, $page, $r)
    {
        $page_details = Page::with('faqs','meta')
            ->where('id', $page)
            ->firstOrFail();

        $packages = Package::where('is_active', 1)
            ->where('country_id', $country)
            ->where('is_deleted', '!=', 1)
            ->with([
                'location' => function ($q) {
                    $q->orderByDesc('id')->with('details');
                },
                'category'
            ])
            ->get();

        // 🔹 Extract and paginate unique locations
        $locations = $packages
            ->pluck('location')
            ->flatten(1)
            ->unique('id')
            ->values();

        $totalLocation = $locations->count();
        $currentPage = (int) $r->get('page', 1);
        $limit       = (int) $r->get('limit', 9);
        $locations   = $locations->forPage($currentPage, $limit)->values();

        // 🔹 Extract unique categories
        $categories = $packages
            ->pluck('category')
            ->unique('id')
            ->values();

        // 🔹 Deal Packages
        $deal_packages = DealPackages::where('page_id', $page)
            ->where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->with(['package.details','package.location.country.continent'])
            ->get()
            ->pluck('package')
            ->filter()
            ->values();

        return response()->json([
            'status'  => 'success',
            'message' => 'details',
            'data'    => [
                'details'       => new PageResource($page_details),
                'locations'     => LocationResource::collection($locations),
                'categories'    => CategoryResource::collection($categories),
                'deal_packages' => PackageResource::collection($deal_packages),
                'pagination'    => [
                    'current_page' => $currentPage,
                    'per_page'     => $limit,
                    'total'  => $totalLocation,
                ],
            ],
        ], 200);
    }


    public function luxuryTrain(Request $r)
    {
        $page_details = Page::with('faqs','meta')
            ->where('id', 2) // train page
            ->firstOrFail();

        return response()->json([
            'status'  => 'success',
            'message' => 'Details',
            'data'    => [
                'details'    => new PageResource($page_details)
            ],
        ], 200);
    }

    public function luxuryHotel(Request $r)
    {
        $page_details = Page::with('faqs','meta')
            ->where('id', 5) // train page
            ->firstOrFail();

        return response()->json([
            'status'  => 'success',
            'message' => 'Details',
            'data'    => [
                'details'    => new PageResource($page_details)
            ],
        ], 200);
    }

    public function car(Request $r)
    {
        $page_details = Page::with('faqs','meta')
            ->where('id', 6)
            ->firstOrFail();

        $cars = Car::where('is_active', 1)
            ->where('is_deleted', '!=', 1)
            ->with('category')
            ->orderBy('id','desc')
            ->get();

        $categories = $cars->pluck('category')->unique('id')->values();

        $fleets = $cars
            ->groupBy('type')
            ->map(function ($group) {
                return $group->take(5)->values();
            });

        $routes = CarRoute::where('is_active', 1)
            ->with('details')
            ->where('is_deleted', '!=', 1)
            ->orderBy('id','desc')
            ->get();

        // 🔹 Apply limit AFTER fetching
        $limitedCars = $cars->take(6); // or $cars->forPage($r->page ?? 1, 10)
        $totalCars = $cars->count();

        return response()->json([
            'status'  => 'success',
            'message' => 'Details',
            'data'    => [
                'details'    => new PageResource($page_details),
                'categories' => CarCategoryResource::collection($categories),
                'cars'       => CarResource::collection($limitedCars),
                'routes'     => CarRouteResource::collection($routes),
                'fleets'     => $fleets->map(fn($group) => CarResource::collection($group)),
                'pagination'  => [
                    'total' => $totalCars,
                    'page'  => 1,
                    'limit' => 6,
                ]
            ],
        ], 200);
    }


    public function customizedHolidays(Request $r)
    {
        $page_details = Page::with('faqs','meta')
            ->where('id', 7) 
            ->firstOrFail();
        return response()->json([
            'status'  => 'success',
            'message' => 'Details',
            'data'    => [
                'details'    => new PageResource($page_details)
            ],
        ], 200);
    }

}
