<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Review;
use App\Models\Package;
use App\Models\Hotel;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ReviewController extends Controller
{
    // ✅ Store review (already done by you)
    public function store(Request $r)
    {
        $r->validate([
            'rating'        => 'required|integer|min:1|max:5',
            'customer_name' => 'required|string',
            'comment'       => 'nullable|string',
            'package_slug'  => 'nullable|exists:packages,slug',
            'hotel_slug'    => 'nullable|exists:hotels,slug',
            'car_slug'      => 'nullable|exists:cars,slug',
        ]);

        $data = [
            'rating'        => $r->rating,
            'comment'       => $r->comment ?? null,
            'customer_name' => $r->customer_name,
        ];

        if ($r->filled('package_slug')) {
            $package = Package::where('slug', $r->package_slug)->first();
            $data['package_id'] = $package->id;
        }

        if ($r->filled('hotel_slug')) {
            $hotel = Hotel::where('slug', $r->hotel_slug)->first();
            $data['hotel_id'] = $hotel->id;
        }

        // if ($r->filled('car_slug')) {
        //     $car = Car::where('slug', $r->car_slug)->first();
        //     $data['car_id'] = $car->id;
        // }

        Review::create($data);

        return response()->json([
            'success' => true,
            'message' => 'Review submitted successfully',
        ]);
    }

    // ✅ Get all approved reviews for all packages
    public function packageReviews()
    {
        $reviews = Review::select('customer_name','rating','comment','created_at')
            ->whereNotNull('package_id')
            ->where('is_approved', 1)
            ->latest()
            ->get()
            ->map(function ($review) {
                return [
                    'customer_name' => $review->customer_name,
                    'rating'        => $review->rating,
                    'comment'       => $review->comment,
                    'days'          => Carbon::parse($review->created_at)->diffForHumans(),
                ];
            });

        return response()->json([
            'success' => true,
            'message' => 'Listing',
            'data' => $reviews
        ]);
    }

    // ✅ Get approved reviews for a single package
    public function packageReviewsBySlug($slug)
    {
        $package = Package::where('slug', $slug)->firstOrFail();

        $reviews = Review::select('customer_name','rating','comment','created_at')
            ->where('package_id', $package->id)
            ->where('is_approved', 1)
            ->latest()
            ->get()
            ->map(function ($review) {
                return [
                    'customer_name' => $review->customer_name,
                    'rating'        => $review->rating,
                    'comment'       => $review->comment,
                    'days'          => Carbon::parse($review->created_at)->diffForHumans(),
                ];
            });

        return response()->json([
            'success' => true,
            'message' => 'Listing',
            'data' => $reviews
        ]);
    }

    // ✅ Get all approved reviews for all hotels
    public function hotelReviews()
    {
        $reviews = Review::select('customer_name','rating','comment','created_at')
            ->whereNotNull('hotel_id')
            ->where('is_approved', 1)
            ->latest()
            ->get()
            ->map(function ($review) {
                return [
                    'customer_name' => $review->customer_name,
                    'rating'        => $review->rating,
                    'comment'       => $review->comment,
                    'days'          => Carbon::parse($review->created_at)->diffForHumans(),
                ];
            });

        return response()->json([
            'success' => true,
            'message' => 'Listing',
            'data' => $reviews
        ]);
    }

    // ✅ Get approved reviews for a single hotel
    public function hotelReviewsBySlug($slug)
    {
        $hotel = Hotel::where('slug', $slug)->firstOrFail();

        $reviews = Review::select('customer_name','rating','comment','created_at')
            ->where('hotel_id', $hotel->id)
            ->where('is_approved', 1)
            ->latest()
            ->get()
            ->map(function ($review) {
                return [
                    'customer_name' => $review->customer_name,
                    'rating'        => $review->rating,
                    'comment'       => $review->comment,
                    'days'          => Carbon::parse($review->created_at)->diffForHumans(),
                ];
            });

        return response()->json([
            'success' => true,
            'message' => 'Listing',
            'data' => $reviews
        ]);
    }

    // ✅ Get all approved reviews for all cars
    public function carReviews()
    {
        $reviews = Review::select('customer_name','rating','comment','created_at')
            ->whereNotNull('car_id')
            ->where('is_approved', 1)
            ->latest()
            ->get()->map(function ($review) {
                return [
                    'customer_name' => $review->customer_name,
                    'rating'        => $review->rating,
                    'comment'       => $review->comment,
                    'days'          => Carbon::parse($review->created_at)->diffForHumans(),
                ];
            });

        return response()->json([
            'success' => true,
            'message' => 'Listing',
            'data' => $reviews
        ]);
    }

    // ✅ Get approved reviews for a single car
    public function carReviewsBySlug($slug)
    {
        $car = Car::where('slug', $slug)->firstOrFail();

        $reviews = Review::select('customer_name','rating','comment','created_at')
            ->where('car_id', $car->id)
            ->where('is_approved', 1)
            ->latest()
            ->get()
            ->map(function ($review) {
                return [
                    'customer_name' => $review->customer_name,
                    'rating'        => $review->rating,
                    'comment'       => $review->comment,
                    'days'          => Carbon::parse($review->created_at)->diffForHumans(),
                ];
            });

        return response()->json([
            'success' => true,
            'message' => 'Listing',
            'data' => $reviews
        ]);
    }
}
