<?php

namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;

use App\Models\Train;
use App\Models\TrainImage;
use App\Models\TrainFaq;
use App\Models\TrainMetaData;
use App\Models\TrainCabin;
use App\Models\TrainCabinImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;

class TrainController extends Controller
{
    
    public function index(Request $r)
    {
        if ($r->exists('cabins')) {
           $cabins = TrainCabin::where('train_id',$r->id)->with('images')->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'cabins'    => $cabins
                ]);
            }
        } else {
            $faqs = TrainFaq::where('train_id',$r->id)->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'faqs'    => $faqs
                ]);
            }
            
        }
    }

    public function store(Request $r)
    {
        try {
            $validatedData = $r->validate([
                'title'             => 'required|string|max:256',
                'primary_image'     => 'required|image|mimes:webp',
                'short_description' => 'required|string',
                'long_description'  => 'required|string',
                'destinations'      => 'required|string',
                'facilities'        => 'required|string'
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation Failed',
                'errors' => $e->errors()
            ], 422);
        }

        $trainData = $r->only([
            'title',
            'short_description',
            'long_description',
            'destinations',
            'facilities'
        ]);

        $trainData['slug'] = Str::slug($r->title);

        if ($r->hasFile('primary_image')) {
            $file = $r->file('primary_image');

            if ($file->isValid()) {
                // keep original filename
                $filename = $file->getClientOriginalName();

                // store in S3 under "trains/"
                $imagePath = $file->storeAs('trains', $filename, 's3');

                $trainData['primary_image'] = $imagePath;
            }
        }
        $trainData['primary_image_alt'] = $r->primary_image_alt;

        $train = Train::create($trainData);

        if ($r->hasFile('images')) {
            foreach ($r->file('images') as $img) {
                if ($img->isValid()) {
                    // keep original filename
                    $filename = $img->getClientOriginalName();

                    // store in S3 under "trains/"
                    $p = $img->storeAs('trains', $filename, 's3');

                    // save to DB
                    TrainImage::create([
                        'train_id'   => $train->id,
                        'image_path' => $p,
                    ]);
                }
            }
        }


        return redirect()->route('admin.page-settings.luxury-train')
        ->with('active_tab','train_list')
        ->with('success','Train created');
    }


    public function show(Train $train)
    {
        $train->load('cabins.images');
        return response()->json($train);
    }

    public function showMeta(Train $train){
        $train->load('meta');
        return response()->json($train);
    }

    public function update(Request $r, Train $train)
    {
        
        if (!$r->exists('status') && !$r->exists('meta_setting')) {
           
             $validatedData = $r->validate([
                'title'             => 'required|string|max:256',
                //'primary_image'     => 'nullable|image|mimes:webp',
                'short_description' => 'required|string',
                'long_description'  => 'required|string',
                'destinations'      => 'required|string',
                'facilities'        => 'required|string'
            ]);
            
            if ($r->hasFile('primary_image')) {
                $file = $r->file('primary_image');

                if ($file->isValid()) {
                    // delete old file if exists
                    if ($train->primary_image && \Storage::disk('s3')->exists($train->primary_image)) {
                        \Storage::disk('s3')->delete($train->primary_image);
                    }

                    // keep original filename
                    $filename = $file->getClientOriginalName();

                    // store in S3 under "trains/"
                    $path = $file->storeAs('trains', $filename, 's3');

                    $validatedData['primary_image'] = $path;
                }
            }
            $validatedData['primary_image_alt'] = $r->primary_image_alt;
            $validatedData['slug'] = Str::slug($r->title);
            $train->update($validatedData);

            if ($r->hasFile('images')) {
                foreach ($r->file('images') as $img) {
                    if ($img->isValid()) {
                        // keep original filename
                        $filename = $img->getClientOriginalName();

                        // store in S3 under "trains/"
                        $p = $img->storeAs('trains', $filename, 's3');

                        // save to DB
                        TrainImage::create([
                            'train_id'   => $train->id,
                            'image_path' => $p,
                        ]);
                    }
                }
            }


        } 
        elseif($r->exists('meta_setting')) {
            
            $r->validate([
                'meta_title'       => 'nullable|string',
                'meta_description' => 'nullable|string',
                'meta_keywords'    => 'nullable|string',
                'h1_heading'       => 'nullable|string',
                'meta_details'     => 'nullable|string',
            ]);
            if ($train->meta) {
                //print_r($r->toArray());die;
                $train->meta->update([
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            } else {
                TrainMetaData::create([
                    'train_id' => $train->id,
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            }
        }
        elseif($r->exists('status')) {
            $train->is_active = $r->status;
            $train->save();
        }

        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Train status updated successfully',
                'data'    => $train
            ]);
        }

        return redirect()->route('admin.page-settings.luxury-train')
            ->with('active_tab','train_list')
            ->with('success','Train updated successfully');
    }

    public function updateFaq(Request $r, Train $train){
        $train->faqs()->delete();
        if ($r->has('faqs')) {
            foreach ($r->faqs as $obj) {
                TrainFaq::create([
                    'train_id'=>$train->id,
                    'question' => $obj['question'],
                    'answer'   => $obj['answer'] ?? null,
                ]);
            }
        }
        return redirect()->route('admin.page-settings.luxury-train')
        ->with('active_tab','train_list')
        ->with('success', 'Train faq updated successfully');
    }

    public function deleteImage(Request $r){
        $img = TrainImage::find($r->id);
        //echo $img->image_path;die;
        if($img){
            // delete file from storage
            if(Storage::disk('s3')->exists($img->image_path)){
                Storage::disk('s3')->delete($img->image_path);
            }
            $img->delete();
            return response()->json(['success'=>true]);
        }
        return response()->json(['success'=>false]);
    }

    public function destroy(Train $train)
    {
        $train->is_deleted = true;
        $train->save();
        if(Storage::disk('s3')->exists($train->primary_image)){
            Storage::disk('s3')->delete($train->primary_image);
        }
        return response()->json(['success'=>true,'message' => 'Train marked as deleted successfully']);
    }

    public function slugDuplicateCheck(Request $r){
        $slug = Str::slug($r->title);
        $slugExists = Train::where('slug', $slug)
        ->when($r->exists('id')? $r->id : '', function ($query, $id) {
            return $query->where('id', '!=', $id);
        })
        ->where('is_deleted', '!=', 1)
        ->exists();
        return response()->json(['exists' => $slugExists]);
    }

    public function updateCabin(Request $request, Train $train)
    {
        // ✅ Delete cabins removed in modal
        if ($request->filled('deleted_cabins')) {
            foreach ($request->deleted_cabins as $cabinId) {
                $cabin = TrainCabin::with('images')->find($cabinId);
                if ($cabin) {
                    // delete related images from storage
                    foreach ($cabin->images as $img) {
                        if (\Storage::disk('s3')->exists($img->image_path)) {
                            \Storage::disk('s3')->delete($img->image_path);
                        }
                        $img->delete();
                    }
                    $cabin->delete();
                }
            }
        }

        // ✅ Update / create cabins
        if ($request->has('cabins')) {
            foreach ($request->cabins as $cabinData) {
                // update existing
                if (!empty($cabinData['id'])) {
                    $cabin = TrainCabin::find($cabinData['id']);
                    if ($cabin) {
                        $cabin->update([
                            'title' => $cabinData['title'],
                        ]);
                    }
                } else {
                    // new cabin
                    $cabin = TrainCabin::create([
                        'train_id' => $train->id,
                        'title'    => $cabinData['title'],
                    ]);
                }

                // handle uploaded images
                if (!empty($cabinData['images'])) {
                    foreach ($cabinData['images'] as $imageFile) {
                        if ($imageFile instanceof \Illuminate\Http\UploadedFile) {
                            // keep original filename
                            $filename = $imageFile->getClientOriginalName();

                            // store in S3 under "cabins/"
                            $path = $imageFile->storeAs('cabins', $filename, 's3');

                            $cabin->images()->create([
                                'image_path' => $path,
                            ]);
                        }
                    }
                }

            }
        }

        return redirect()
            ->route('admin.page-settings.luxury-train')
            ->with('active_tab', 'train_list')
            ->with('success', 'Train cabins updated successfully');
    }


    public function deleteCabinImage(Request $r)
    {
        $image = TrainCabinImage::findOrFail($r->id);

        // delete file from storage
        if (\Storage::disk('s3')->exists($image->image_path)) {
            \Storage::disk('s3')->delete($image->image_path);
        }

        // delete DB record
        $image->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Image deleted successfully',
        ]);
    }
}
