<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ContinentCountry;
use App\Models\Continent;
use App\Models\Country;
use App\Models\Location;
use App\Models\LocationDetails;
use App\Models\LocationFaq;
use App\Models\LocationMetaData;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;


class LocationController extends Controller{

    public function index(Request $r){
        if ($r->exists('country_id') && !$r->exists('faqs')) {
            return response()->json(['cities'=>Location::where('country_id',$r->country_id)->get()]);

        }
        elseif(!$r->exists('country_id') && $r->exists('faqs')){
            $faqs = LocationFaq::where('location_id',$r->id)->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'faqs'    => $faqs
                ]);
            }
        } else {
            $continents = Continent::all();
            $countries = Country::with('continent')
                ->orderByRaw("
                    CASE 
                        WHEN code = 'IN' THEN 1
                        WHEN code = 'LK' THEN 2
                        ELSE 3
                    END
                ")
                ->orderBy('id')
                ->paginate(list_config()['limit'], ['*'], 'country_page');
            $all_countries = Country::with('continent')
                ->orderByRaw("
                    CASE 
                        WHEN code = 'IN' THEN 1
                        WHEN code = 'LK' THEN 2
                        ELSE 3
                    END
                ")
                ->orderBy('id')
                ->get();
            $locations = Location::with('country.continent')->paginate(list_config()['limit'], ['*'], 'location_page');

            // If you need distinct continent names & codes
            $continents_master = ContinentCountry::select('continent_name as name','continent_code as code')->distinct()->get();

            return view('admin.locations.index', compact(
                'continents',
                'countries',
                'locations',
                'continents_master',
                'all_countries'
            ));
        }

    }

    public function store(Request $request){
        $request->validate([
            'country_id' => 'required|exists:countries,id',
            'name' => 'required|string|max:255'
        ]);

        $slug = Str::slug($request->name.'-tour-packages');

        if (Location::where('slug', $slug)->exists()) {
            throw ValidationException::withMessages([
                'name' => 'City already exists.',
            ]);
        }
        $location = Location::create([
            'country_id' => $request->country_id,
            'name' => $request->name,
            'slug' => $slug,
            'is_top_trending' => $request->is_top_trending ? 1 : 0
        ]);

        return response()->json($location->load('country'));
    }

    public function show(Location $location){

        $location->load('country.continent','details');
        return response()->json($location);
    }

    public function showMeta(Location $location){
        $location->load('meta');
        return response()->json($location);
    }

    public function update(Request $r, Location $location){
        // echo '<pre>';
        // print_r($r->toArray());die;
        if(!$r->exists('page_setting') && !$r->exists('meta_setting')){
            //echo 'hjfghjb';die;
            $r->validate([
                'country_id' => 'required|exists:countries,id',
                'name' => 'required|string|max:255'
            ]);

            $slug = Str::slug($r->name.'-tour-packages');
            $slug_exists = Location::where('slug', $slug)
            ->when($r->exists('id')? $r->id : '', function ($query, $id) {
                return $query->where('id', '!=', $id);
            })
            ->exists();
            if ($slug_exists) {
                throw ValidationException::withMessages([
                    'name' => 'City already exists.',
                ]);
            }
            $location->update([
                'name'        => $r->name,
                'country_id' => $r->country_id,
                'slug'        => $slug,
                'is_top_trending' => $r->is_top_trending ? 1 : 0
            ]);
        } elseif($r->exists('page_setting')) {
            $r->validate([
                'banner_image' => 'nullable|image|mimes:webp'
            ]);
            $path = $location->details->banner_image ?? '';
            if ($r->hasFile('banner_image')) {
                $img = $r->file('banner_image');
                if ($img->isValid()) {
                    if ($location->details && $location->details->banner_image && $location->details->banner_image!=0 && \Storage::disk('s3')->exists($location->details->banner_image)) {
                        \Storage::disk('s3')->delete($location->details->banner_image);
                    }

                    // keep original filename
                    $filename = $img->getClientOriginalName();

                    // store in S3 under "location/"
                    $path = $img->storeAs('location', $filename, 's3');
                }
            }

            if ($location->details) {
                $location->details->update([
                    'title' => $r->title,
                    'sub_title' => $r->sub_title,
                    'banner_image' => $path != '0' ? $path : '',
                    'banner_image_alt'=>$r->banner_image_alt,
                    'about' => $r->about
                ]);
            } else {
                LocationDetails::create([
                    'location_id' => $location->id,
                    'title' => $r->title,
                    'sub_title' => $r->sub_title,
                    'banner_image' => $path != '0' ? $path : '',
                    'banner_image_alt'=>$r->banner_image_alt,
                    'about' => $r->about
                ]);
            }
        } elseif($r->exists('meta_setting')) {
            
            $r->validate([
                'meta_title'       => 'nullable|string',
                'meta_description' => 'nullable|string',
                'meta_keywords'    => 'nullable|string',
                'h1_heading'       => 'nullable|string',
                'meta_details'     => 'nullable|string',
            ]);
            if ($location->meta) {
                //print_r($r->toArray());die;
                $location->meta->update([
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            } else {
                LocationMetaData::create([
                    'location_id' => $location->id,
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            }
        }
        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Location updated successfully',
                'data'    => $location
            ]);
        }
        return redirect()->route('admin.locations.index')->with('success', 'Location updated successfully');
    }

    public function updateFaq(Request $r, Location $location){
        $location->faqs()->delete();
        if ($r->has('faqs')) {
            foreach ($r->faqs as $obj) {
                LocationFaq::create([
                    'location_id'=>$location->id,
                    'question' => $obj['question'],
                    'answer'   => $obj['answer'] ?? null,
                ]);
            }
        }
        return redirect()->route('admin.locations.index')->with('success', 'Location faq updated successfully');
    }

}
