<?php

namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;

use App\Models\Hotel;
use App\Models\HotelImage;
use App\Models\HotelFaq;
use App\Models\HotelRoom;
use App\Models\HotelRoomImage;
use App\Models\HotelToDo;
use App\Models\HotelMetaData;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;

class HotelController extends Controller
{
    
    public function index(Request $r)
    {
        if ($r->exists('rooms')) {
           $cabins = HotelRoom::where('hotel_id',$r->id)->with('images')->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'rooms'    => $cabins
                ]);
            }
        }elseif ($r->exists('todos')) {
           $todos = HotelToDo::where('hotel_id',$r->id)->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'todos'    => $todos
                ]);
            }
        }
         else {
            $faqs = HotelFaq::where('hotel_id',$r->id)->get();
            if ($r->ajax()) {
                return response()->json([
                    'status'  => 'success',
                    'faqs'    => $faqs
                ]);
            }
            
        }
    }

    public function store(Request $r)
    {
        try {
            $validatedData = $r->validate([
                'title'             => 'required|string|max:256',
                'location'          => 'required|string|max:256',
                'primary_image'     => 'required|image|mimes:webp',
                'short_description' => 'required|string',
                'refund_cancellation_policy'=> 'required|string'
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation Failed',
                'errors' => $e->errors()
            ], 422);
        }

        $hotelData = $r->only([
            'title',
            'location',
            'short_description',
            'refund_cancellation_policy',
            'facilities',
            'is_top_luxury'
        ]);

        $hotelData['slug'] = Str::slug($r->title);

        if ($r->hasFile('primary_image')) {
            $file = $r->file('primary_image');

            if ($file->isValid()) {
                // keep original filename
                $filename = $file->getClientOriginalName();

                // store in S3 with original filename inside "hotels" folder
                $imagePath = $file->storeAs('hotels', $filename, 's3');

                $hotelData['primary_image'] = $imagePath;
            }
        }
        $hotelData['primary_image_alt'] = $r->primary_image_alt;

        $hotelData['facilities'] = $hotelData['facilities'] ?? [];
        $hotelData['is_top_luxury'] = $hotelData['is_top_luxury'] ? 1 : 0;
        $hotel = Hotel::create($hotelData);

        if ($r->hasFile('images')) {
            foreach ($r->file('images') as $img) {
                if ($img->isValid()) {
                    // get original filename
                    $filename = $img->getClientOriginalName();

                    // store with original filename in S3 inside "hotel_rooms"
                    $p = $img->storeAs('hotel_rooms', $filename, 's3');

                    // save to DB
                    HotelImage::create([
                        'hotel_id'   => $hotel->id,
                        'image_path' => $p,
                        'image_alt' => $r->images_alt[$key] ?? null,
                    ]);
                }
            }
        }


        return redirect()->route('admin.page-settings.luxury-hotel')
        ->with('active_tab','hotel_list')
        ->with('success','Hotel created');
    }

    public function showMeta(Hotel $hotel){
        $hotel->load('meta');
        return response()->json($hotel);
    }

    public function show(Hotel $hotel)
    {
        $hotel->load('images','rooms.images');
        return response()->json($hotel);
    }

    public function update(Request $r, Hotel $hotel)
    {
        
        if (!$r->exists('status') && !$r->exists('meta_setting')) {
           
             $validatedData = $r->validate([
                'title'             => 'required|string|max:256',
                'location'          => 'required|string|max:256',
                'short_description' => 'required|string',
                'refund_cancellation_policy'=> 'required|string',
                'primary_image'      => 'nullable|image|mimes:webp',
                'images'             => 'nullable',
                'images.*'           => 'nullable|image|mimes:webp',
            ]);
            
            if ($r->hasFile('primary_image')) {
                // delete old image if exists
                if ($hotel->primary_image && \Storage::disk('s3')->exists($hotel->primary_image)) {
                    \Storage::disk('s3')->delete($hotel->primary_image);
                }

                $file = $r->file('primary_image');
                $filename = $file->getClientOriginalName(); // original file name

                // save with original filename inside "hotels" folder
                $path = $file->storeAs('hotels', $filename, 's3');

                $validatedData['primary_image'] = $path;
            }
            $validatedData['primary_image_alt'] = $r->primary_image_alt;


            // if ($r->hasFile('primary_image')) {
            //     if ($hotel->primary_image && \Storage::disk('s3')->exists($hotel->primary_image)) {
            //         \Storage::disk('s3')->delete($hotel->primary_image);
            //     }

            //     $filename = time().'_'.$r->file('primary_image')->getClientOriginalName();
            //     $path = $r->file('primary_image')->storeAs('hotels', $filename, 's3');

            //     $validatedData['primary_image'] = $path;
            // }

            $validatedData['slug'] = Str::slug($r->title);
            $validatedData['facilities'] = $r->facilities ?? [];
            $validatedData['is_top_luxury'] = $r->is_top_luxury ? 1 : 0;
            $hotel->update($validatedData);

            if ($r->hasFile('images')) {
                foreach ($r->file('images') as $img) {
                    if ($img->isValid()) {
                        $filename = $img->getClientOriginalName(); // original file name

                        // save with original name inside "hotels" folder
                        $p = $img->storeAs('hotels', $filename, 's3');

                        HotelImage::create([
                            'hotel_id'   => $hotel->id,
                            'image_path' => $p,
                            'image_alt'  => $r->images_alt[$key] ?? null,
                        ]);
                    }
                }
            }

            if ($r->has('existing_images_alt')) {
                foreach ($r->existing_images_alt as $id => $alt) {
                    HotelImage::where('id', $id)->update([
                        'image_alt' => $alt,
                    ]);
                }
            }


        }
        elseif($r->exists('meta_setting')) {
            
            $r->validate([
                'meta_title'       => 'nullable|string',
                'meta_description' => 'nullable|string',
                'meta_keywords'    => 'nullable|string',
                'h1_heading'       => 'nullable|string',
                'meta_details'     => 'nullable|string',
            ]);
            if ($hotel->meta) {
                //print_r($r->toArray());die;
                $hotel->meta->update([
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            } else {
                HotelMetaData::create([
                    'hotel_id' => $hotel->id,
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            }
        }
        elseif($r->exists('status')) {
            $hotel->is_active = $r->status;
            $hotel->save();
        }

        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Hotel status updated successfully',
                'data'    => $hotel
            ]);
        }

        return redirect()->route('admin.page-settings.luxury-hotel')
            ->with('active_tab','hotel_list')
            ->with('success','Hotel updated successfully');
    }

    public function updateFaq(Request $r, Hotel $hotel){
        $hotel->faqs()->delete();
        if ($r->has('faqs')) {
            foreach ($r->faqs as $obj) {
                HotelFaq::create([
                    'hotel_id' => $hotel->id,
                    'question' => $obj['question'],
                    'answer'   => $obj['answer'] ?? null,
                ]);
            }
        }
        return redirect()->route('admin.page-settings.luxury-hotel')
        ->with('active_tab','hotel_list')
        ->with('success', 'Hotel faq updated successfully');
    }

    public function deleteImage(Request $r){
        $img = HotelImage::find($r->id);
        //echo $img->image_path;die;
        if($img){
            // delete file from storage
            if(Storage::disk('s3')->exists($img->image_path)){
                Storage::disk('s3')->delete($img->image_path);
            }
            $img->delete();
            return response()->json(['success'=>true]);
        }
        return response()->json(['success'=>false]);
    }

    public function destroy(Hotel $hotel)
    {
        $hotel->is_deleted = true;
        $hotel->save();

        return response()->json(['success'=>true,'message' => 'Hotel marked as deleted successfully']);
    }

    public function slugDuplicateCheck(Request $r){
        $slug = Str::slug($r->title);
        $slugExists = Hotel::where('slug', $slug)
        ->when($r->exists('id')? $r->id : '', function ($query, $id) {
            return $query->where('id', '!=', $id);
        })
        ->where('is_deleted', '!=', 1)
        ->exists();
        return response()->json(['exists' => $slugExists]);
    }

    public function updateRoom(Request $request, Hotel $hotel)
    {
        // ✅ Delete cabins removed in modal
        if ($request->filled('deleted_rooms')) {
            foreach ($request->deleted_rooms as $roomId) {
                $room = HotelRoom::with('images')->find($roomId);
                if ($room) {
                    // delete related images from storage
                    foreach ($room->images as $img) {
                        if (\Storage::disk('s3')->exists($img->image_path)) {
                            \Storage::disk('s3')->delete($img->image_path);
                        }
                        $img->delete();
                    }
                    $room->delete();
                }
            }
        }

        // ✅ Update / create cabins
        if ($request->has('rooms')) {
            foreach ($request->rooms as $roomData) {
                // update existing
                if (!empty($roomData['id'])) {
                    $room = HotelRoom::find($roomData['id']);
                    if ($room) {
                        $room->update([
                            'title' => $roomData['title'],
                            'details' => $roomData['details'],
                        ]);
                    }
                } else {
                    // new cabin
                    $room = HotelRoom::create([
                        'hotel_id' => $hotel->id,
                        'title'    => $roomData['title'],
                        'details'  => $roomData['details'],
                    ]);
                }

                // handle uploaded images
                if (!empty($roomData['images'])) {
                    foreach ($roomData['images'] as $imageFile) {
                        if ($imageFile instanceof \Illuminate\Http\UploadedFile) {
                            // keep original file name
                            $filename = $imageFile->getClientOriginalName();

                            // store in S3 under hotel_rooms/
                            $path = $imageFile->storeAs('hotel_rooms', $filename, 's3');

                            $room->images()->create([
                                'image_path' => $path,
                            ]);
                        }
                    }
                }

            }
        }

        return redirect()
            ->route('admin.page-settings.luxury-hotel')
            ->with('active_tab', 'hotel_list')
            ->with('success', 'Hotel room updated successfully');
    }


    public function deleteRoomImage(Request $r)
    {
        $image = HotelRoomImage::findOrFail($r->id);

        // delete file from storage
        if (\Storage::disk('s3')->exists($image->image_path)) {
            \Storage::disk('s3')->delete($image->image_path);
        }

        // delete DB record
        $image->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Image deleted successfully',
        ]);
    }

    public function updateTodo(Request $request, Hotel $hotel)
    {
        // ✅ Delete cabins removed in modal
        if ($request->filled('deleted_todos')) {
            foreach ($request->deleted_todos as $todoId) {
                $todo = HotelToDo::find($todoId);
                if ($todo) {
                    if ($todo->banner_image && \Storage::disk('s3')->exists($todo->banner_image)) {
                        \Storage::disk('s3')->delete($todo->banner_image);
                    }
                    $todo->delete();
                }
            }
        }

        // ✅ Update / create todo
        if ($request->has('todos')) {
            foreach ($request->todos as $todoData) {
                // update existing
                if (!empty($todoData['id'])) {
                    $todo = HotelToDo::find($todoData['id']);
                    if ($todo) {
                        $updateData = [
                            'title'   => $todoData['title'],
                            'details' => $todoData['details'],
                        ];

                        // ✅ Handle banner image update
                        if (isset($todoData['banner_image']) && $todoData['banner_image'] instanceof \Illuminate\Http\UploadedFile) {
                            // delete old if exists
                            if ($todo->banner_image && \Storage::disk('s3')->exists($todo->banner_image)) {
                                \Storage::disk('s3')->delete($todo->banner_image);
                            }

                            // keep original file name
                            $filename = $todoData['banner_image']->getClientOriginalName();

                            // store in S3 under hotel_todos/
                            $path = $todoData['banner_image']->storeAs('hotel_todos', $filename, 's3');

                            $updateData['banner_image'] = $path;
                        }


                        $todo->update($updateData);
                    }
                } else {
                    // new todo
                    $createData = [
                        'hotel_id' => $hotel->id,
                        'title'    => $todoData['title'],
                        'details'  => $todoData['details'],
                    ];

                    // ✅ Handle banner image for new todo
                    if (isset($todoData['banner_image']) && $todoData['banner_image'] instanceof \Illuminate\Http\UploadedFile) {
                        // keep original file name
                        $filename = $todoData['banner_image']->getClientOriginalName();

                        // store in S3 under hotel_todos/
                        $path = $todoData['banner_image']->storeAs('hotel_todos', $filename, 's3');

                        $createData['banner_image'] = $path;
                    }


                    HotelToDo::create($createData);
                }
            }
        }

        return redirect()
            ->route('admin.page-settings.luxury-hotel')
            ->with('active_tab', 'hotel_list')
            ->with('success', 'Hotel nearby top to-do things updated successfully');
    }

}
