<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use App\Models\Car;
use App\Models\CarRoute;
use App\Models\CarRouteDetails;
use App\Models\CarRouteMetaData;
use App\Models\CarRoutePageDetails;
use App\Models\CarRouteFaq;

class CarRouteController extends Controller
{
    /**
     * Show all categories
     */
    public function index()
    {
        return response()->json(CarRoute::all());
    }

    /**
     * Store (Add) new category
     */
    public function store(Request $r)
    {
        $r->validate([
            'from_location' => 'required',
            'to_location' => 'required',
        ]);

        $slug = Str::slug($r->from_location.'-to-'.$r->to_location);

        if (CarRoute::where('slug', $slug)->exists()) {
            throw ValidationException::withMessages([
                'name' => 'Route already exists.',
            ]);
        }

        $carRoutes = CarRoute::create([
            'slug' => $slug,
            'from_location' => $r->from_location,
            'to_location' => $r->to_location,
        ]);

        return redirect()->route('admin.page-settings.car')
        ->with('active_tab','route_list')
        ->with('success','Route created');
    }

    public function showMeta(CarRoute $car_route){
        $car_route->load('meta');
        return response()->json($car_route);
    }

    public function showPage(CarRoute $car_route){
        $car_route->load('details');
        return response()->json($car_route);
    }

    /**
     * Show single category
     */
    public function show($id)
    {
        $carRoute = CarRoute::with(['cars.category'])->findOrFail($id);
        return response()->json($carRoute);
    }

    /**
     * Update category
     */
    public function update(Request $r, CarRoute $car_route)
    {
        if ($r->has('status')) {
            $car_route->is_active = $r->status;
            $car_route->save();

            return response()->json([
                'success' => true,
                'message' => 'Status updated successfully',
                'status'  => $car_route->is_active
            ]);
        }

        if($r->exists('meta_setting')) {
            
            $r->validate([
                'meta_title'       => 'nullable|string',
                'meta_description' => 'nullable|string',
                'meta_keywords'    => 'nullable|string',
                'h1_heading'       => 'nullable|string',
                'meta_details'     => 'nullable|string',
            ]);
            if ($car_route->meta) {
                //print_r($r->toArray());die;
                $car_route->meta->update([
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            } else {
                CarRouteMetaData::create([
                    'route_id' => $car_route->id,
                    'meta_title' => $r->meta_title,
                    'meta_description' => $r->meta_description,
                    'meta_keywords' => $r->meta_keywords,
                    'h1_heading' => $r->h1_heading,
                    'meta_details' => $r->meta_details,
                ]);
            }
            return redirect()->route('admin.page-settings.car')
            ->with('active_tab','route_list')
            ->with('success','Route meta updated successfully');
        }

        if ($r->exists('page_setting')) {
            // ✅ Validation
            $validatedData = $r->validate([
                'title'         => 'required|string|max:255',
                'description'   => 'required|string',
                'banner_image'  => 'sometimes|image|mimes:webp', // optional: file size limit
                'banner_image_alt' => 'nullable|string|max:255'
            ]);

            // ✅ Get existing image path safely
            $path = $car_route->details->banner_image ?? '';

            // ✅ Handle image upload
            if ($r->hasFile('banner_image')) {
                $img = $r->file('banner_image');

                if ($img->isValid()) {
                    // ✅ Delete old image if it exists
                    if (!empty($car_route->details->banner_image) &&
                        \Storage::disk('s3')->exists($car_route->details->banner_image)) {
                        \Storage::disk('s3')->delete($car_route->details->banner_image);
                    }

                    $filename = $img->getClientOriginalName();

                    // ✅ Store file in S3 under "car_route" directory
                    $path = $img->storeAs('car_route', $filename, 's3');
                }
            }

            // ✅ Prepare update data
            $obj = [
                'route_id'         => $car_route->id,
                'title'            => $r->title,
                'description'      => $r->description,
                'banner_image'     => $path,
                'banner_image_alt' => $r->banner_image_alt,
            ];

            if ($car_route->details) {
                //print_r($r->toArray());die;
                $car_route->details->update($obj);
            } else {
                CarRoutePageDetails::create($obj);
            }

            // ✅ Redirect with success message
            return redirect()
                ->route('admin.page-settings.car')
                ->with('active_tab', 'route_list')
                ->with('success', 'Route meta updated successfully');
        }


        $r->validate([
            'from_location' => 'required',
            'to_location' => 'required',
        ]);

        $slug = Str::slug($r->from_location.'-to-'.$r->to_location);

        if (CarRoute::where('slug', $slug)->where('id', '!=', $car_route->id)->exists()) {
            throw ValidationException::withMessages([
                'name' => 'Route already exists.',
            ]);
        }

        $car_route->update([
            'slug' => $slug,
            'from_location' => $r->from_location,
            'to_location' => $r->to_location,
        ]);

        return redirect()->route('admin.page-settings.car')
        ->with('active_tab','route_list')
        ->with('success','Route updated successfully');
    }

    /**
     * Delete category
     */

    public function destroy(CarRoute $car_route)
    {
        $car_route->is_deleted = true;
        $car_route->save();

        return response()->json(['success'=>true,'message' => 'Route deleted successfully']);
    }

    public function slugDuplicateCheck(Request $r){ 
        return response()->json([
            'exists' => CarRoute::where('slug', Str::slug($r->from_location.'-to-'.$r->to_location))
            ->where('id', '!=', $r->id)
            ->where('is_deleted', '!=', 1)
            ->exists()
        ]);

    }

    public function getRouteCars($id)
    {
        $all_car = Car::with(['category'])->where('is_active', 1)->where('is_deleted', '!=', 1)->get();
        $carRoute = CarRoute::with(['cars'])->findOrFail($id);
        return response()->json(['route'=>$carRoute,'all_car'=>$all_car]);
    }

    public function syncCars(Request $r)
    {
        // echo '<pre>';
        // print_r($r->toArray());die;
        $r->validate([
            'route_id' => 'required|integer|exists:car_routes,id',
            'car_ids'  => 'array', // can be empty if user unchecks all
            'car_ids.*' => 'integer|exists:cars,id',
        ]);

        $routeId = $r->route_id;
        $carIds  = $r->car_ids ?? []; // selected cars (checked)

        // 🔄 Sync logic:
        // 1️⃣ Delete cars not in the list.
        CarRouteDetails::where('route_id', $routeId)
            ->whereNotIn('car_id', $carIds)
            ->delete();

        // 2️⃣ Add new cars that don’t already exist.
        $existingCarIds = CarRouteDetails::where('route_id', $routeId)
            ->pluck('car_id')
            ->toArray();

        $newCarIds = array_diff($carIds, $existingCarIds);

        foreach ($newCarIds as $carId) {
            CarRouteDetails::create([
                'route_id' => $routeId,
                'car_id'   => $carId,
            ]);
        }

        return redirect()->route('admin.page-settings.car')
        ->with('active_tab','route_list')
        ->with('success','Route car sync successfully');
    }

    public function faqs(Request $r){
        $faqs = CarRouteFaq::where('route_id',$r->id)->get();
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'faqs'    => $faqs
            ]);
        }
    }

    public function updateFaq(Request $r, CarRoute $car_route){
        $car_route->faqs()->delete();
        if ($r->has('faqs')) {
            foreach ($r->faqs as $obj) {
                CarRouteFaq::create([
                    'route_id'  => $car_route->id,
                    'question' => $obj['question'],
                    'answer'   => $obj['answer'] ?? null,
                ]);
            }
        }
        return redirect()
        ->route('admin.page-settings.car')
        ->with('active_tab','route_list')
        ->with('success', 'Faq updated successfully');
    }

}
