<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Car;
use Illuminate\Support\Str;

class CarController extends Controller {

    public function index(Request $r){ 
        if ($r->exists('id')) {
            return response()->json(['car'=> Car::where('is_deleted','!=',1)->where('id',$r->id)->first()]);
        }
    }

    public function store(Request $r){
        // echo '<pre>';
        // print_r($r->toArray()); die;
        $validated = $r->validate([
            'title'       => 'required|max:256',
            'category_id' => 'required|exists:car_categories,id',
            'type'        => 'required|max:256',
            'seats'        => 'required',
            'primary_image'=> 'required|image|mimes:webp',
            'primary_image_alt' => 'required|max:256'
        ]);
        $path = '';
        if ($r->hasFile('primary_image')) {
            $path = $r->file('primary_image')->store('cars', 's3');
        }
        $car = Car::create([
            'title'        => $validated['title'],
            'slug'        => Str::slug($validated['title']),
            'category_id'       => $validated['category_id'],
            'type'   => $validated['type'],
            'primary_image'=> $path,
            'primary_image_alt' => $validated['primary_image_alt'],
            'seats' => $validated['seats']
        ]);

        if ($r->ajax()) {
            return response()->json([
                'success'  => true,
                'message'  => 'Car created successfully',
                'car' => $car
            ]);
        }
        // Fallback for non-AJAX
        return redirect()
        ->route('admin.page-settings.car')
        ->with('active_tab','car_list')
        ->with('success', 'Car created successfully');
    }


    public function edit(Car $car){
        if (request()->ajax()) {
            return response()->json($car);
        }
    }

    public function show(Car $car)
    {
        return response()->json($car);
    }


    public function update(Request $r, Car $car){

        if (!$r->exists('status')) {
            $r->validate([
                'title'       => 'required|max:256',
                'category_id' => 'required|exists:car_categories,id',
                'type'        => 'required|max:256',
                'seats'        => 'required',
                'primary_image'=> 'nullable|image|mimes:webp',
                'primary_image_alt' => 'required|max:256'
            ]);
            $path = $car->primary_image ?? '';
            if ($r->hasFile('primary_image')) {
                // delete old file if it exists in S3
                if ($car->primary_image && \Storage::disk('s3')->exists($car->primary_image)) {
                    \Storage::disk('s3')->delete($car->primary_image);
                }

                $file = $r->file('primary_image');
                $filename = $file->getClientOriginalName(); // keep original filename

                // store in S3 under cars folder
                $path = $file->storeAs('cars', $filename, 's3');
            }


            $car->update([
                'title'       => $r->title,
                'slug'        => Str::slug($r->title),
                'category_id'   => $r->category_id,
                'primary_image'=> $path,
                'primary_image_alt' => $r->primary_image_alt,
                'type' => $r->type,
                'seats' => $r->seats,
                
            ]);
        } else {
            $car->is_active = $r->status;
            $car->save();
        }

        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Car updated successfully',
                'data'    => $car
            ]);
        }

        // Fallback for non-AJAX
        return redirect()
        ->route('admin.page-settings.car')
        ->with('active_tab','car_list')
        ->with('success', 'Car updated successfully');
    }


    public function destroy(Car $car){ 
        $car->is_deleted = 1; 
        $car->save();
        if ($car->primary_image && \Storage::disk('s3')->exists($car->primary_image)) {
            \Storage::disk('s3')->delete($car->primary_image);
        }
        return response()->json(['success'=>true]);
    }

    public function slugDuplicateCheck(Request $r){ 
        return response()->json([
            'exists' => Car::where('slug',  Str::slug($r->title))
            ->where('id', '!=', $r->id)
            ->where('is_deleted', '!=', 1)
            ->exists()
        ]);

    }
}
