<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use App\Models\CarCategory;

class CarCategoryController extends Controller
{
    /**
     * Show all categories
     */
    public function index()
    {
        return response()->json(CarCategory::all());
    }

    /**
     * Store (Add) new category
     */
    public function store(Request $r)
    {
        $r->validate([
            'name' => 'required|string|max:255',
        ]);

        $slug = Str::slug($r->name);

        if (CarCategory::where('slug', $slug)->exists()) {
            throw ValidationException::withMessages([
                'name' => 'Category already exists.',
            ]);
        }

        $category = CarCategory::create([
            'name' => $r->name,
            'slug' => $slug,
        ]);

        return redirect()->route('admin.page-settings.car')
        ->with('active_tab','category_list')
        ->with('success','Category created');
    }

    /**
     * Show single category
     */
    public function show($id)
    {
        $category = CarCategory::findOrFail($id);
        return response()->json($category);
    }

    /**
     * Update category
     */
    public function update(Request $r, CarCategory $car_category)
    {
        if ($r->has('status')) {
            $car_category->is_active = $r->status;
            $car_category->save();

            return response()->json([
                'success' => true,
                'message' => 'Status updated successfully',
                'status'  => $car_category->is_active
            ]);
        }
        $r->validate([
            'name' => 'required|string|max:255',
        ]);

        $slug = Str::slug($r->name);

        if (CarCategory::where('slug', $slug)->where('id', '!=', $car_category->id)->exists()) {
            throw ValidationException::withMessages([
                'name' => 'Category already exists.',
            ]);
        }

        $car_category->update([
            'name' => $r->name,
            'slug' => $slug,
        ]);

        return redirect()->route('admin.page-settings.car')
        ->with('active_tab','category_list')
        ->with('success','Category updated successfully');
    }

    /**
     * Delete category
     */

    public function destroy(CarCategory $car_category)
    {
        $car_category->is_deleted = true;
        $car_category->save();

        return response()->json(['success'=>true,'message' => 'Category deleted successfully']);
    }

    public function slugDuplicateCheck(Request $r){ 
        return response()->json([
            'exists' => CarCategory::where('slug', Str::slug($r->name))
            ->where('id', '!=', $r->id)
            ->where('is_deleted', '!=', 1)
            ->exists()
        ]);

    }
}
