<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Banner;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;


class BannerController extends Controller{

    public function store(Request $r){
        $r->validate([
            'package_id' => 'required|exists:packages,id',
            'title' => 'required|string|max:255',
            'banner_image' => 'required|image|mimes:webp'
        ]);

        $obj = [
            'package_id' => $r->package_id,
            'title' => $r->title,
            'banner_image_alt' => $r->banner_image_alt,
        ];
        session()->flash('active_tab', 'banner');
        if($r->exists('is_static')){
            Banner::where('is_static', 1)->update(['is_active' => 0]);
            $obj['is_static'] = 1;
            session()->flash('active_tab', 'sbanner');
        }

        if ($r->hasFile('banner_image')) {
            $file = $r->file('banner_image');

            // get original filename
            $filename = $file->getClientOriginalName();

            // store in S3 with original filename
            $path = $file->storeAs('banner', $filename, 's3');

            $obj['banner_image'] = $path;
        }


        $banner = Banner::create($obj);

        return response()->json(['success'=>true]);
    }

    public function edit(Banner $banner){
        $banner = Banner::where('id',$banner->id)->first();
        return response()->json($banner);
    }

    public function update(Request $r, Banner $banner){

        if (!$r->exists('status')) {
            $r->validate([
                'title'        => 'required|max:150',
                'package_id'   => 'required|exists:packages,id',
                'banner_image' => 'nullable|image|mimes:webp'
            ]);
            $path = $banner->banner_image ?? '';
            if ($r->hasFile('banner_image')) {
                // delete old file if it exists in S3
                if ($banner->banner_image && \Storage::disk('s3')->exists($banner->banner_image)) {
                    \Storage::disk('s3')->delete($banner->banner_image);
                }

                $file = $r->file('banner_image');
                $filename = $file->getClientOriginalName(); // original name with extension

                // store with original filename in S3
                $path = $file->storeAs('banner', $filename, 's3');

                $banner->banner_image = $path; // or $obj['banner_image'] = $path;
            }

            $obj = [
                'title'       => $r->title,
                'package_id'   => $r->package_id,
                'banner_image'=> $path,
                'banner_image_alt' => $r->banner_image_alt,
            ];
            session()->flash('active_tab', 'banner');
            if($r->exists('is_static')){
                $obj['is_static'] = 1;
                session()->flash('active_tab', 'sbanner');
            }

            $banner->update($obj);
            if ($banner->is_static) {
                session()->flash('active_tab', 'sbanner');
            }
        } else {
            if ($banner->is_static) {
                session()->flash('active_tab', 'sbanner');
                if ($r->status) {
                    // If activating this static banner → deactivate others
                    Banner::where('is_static', 1)->update(['is_active' => 0]);
                    $banner->is_active = 1;
                } else {
                    // If trying to deactivate → check if another static active exists
                    $activeStatic = Banner::where('is_static', 1)
                    ->where('is_active', 1)
                    ->where('id', '!=', $banner->id)
                    ->exists();

                    if (!$activeStatic) {
                        // No other static is active → force this one to stay active
                        $banner->is_active = 1;
                        return response()->json([
                            'status'  => 'error',
                            'message' => 'At least one banner need to active',
                            'data'    => $banner
                        ]);
                    } else {
                        // Safe to deactivate
                        $banner->is_active = 0;
                    }
                }
            } else {
                // Normal banner just update status
                $banner->is_active = $r->status ? 1 : 0;
            }

            $banner->save();

        }

        // If AJAX request, return JSON response
        if ($r->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => 'Banner updated successfully',
                'data'    => $banner
            ]);
        }

        // Fallback for non-AJAX
        return redirect()
        ->route('admin.settings.index')
        ->with('success', 'Banner updated successfully');
    }

    public function destroy(Banner $banner){
        $banner->is_deleted = 1; 
        $banner->save();
        if (\Storage::disk('s3')->exists($banner->banner_image)) {
            \Storage::disk('s3')->delete($banner->banner_image);
        }
        return response()->json(['success'=>true]);
    }
}
